<?php
/**
 * Policy Wallet Manager Admin Page
 * Allows admin to generate and manage secure policy wallets for minting
 */

if (!defined('ABSPATH')) {
    exit;
}

use CardanoMintPay\Models\MintModel;

// Get current network setting
$network = get_option('cardano-mint-networkenvironment', 'preprod');

// Get existing ACTIVE policy wallet for this network
$existing_wallet = MintModel::getActivePolicyWallet($network);

// Get archived wallets
$archived_wallets = MintModel::getArchivedPolicyWallets($network);
$archived_count = MintModel::countArchivedPolicyWallets($network);

// Check if we just created a wallet and need to display the mnemonic
$show_mnemonic = isset($_GET['created']) && $_GET['created'] == '1';
$mnemonic = $show_mnemonic ? get_transient('cardano_policy_wallet_mnemonic_' . get_current_user_id()) : false;

// Delete the transient after retrieving it (ONE-TIME view)
if ($mnemonic) {
    delete_transient('cardano_policy_wallet_mnemonic_' . get_current_user_id());
}

// Decrypt extended key for display (only if admin is viewing their own wallet)
$extended_key_hex = null;
if ($existing_wallet && current_user_can('manage_options')) {
    $extended_key_hex = \CardanoMintPay\Helpers\EncryptionHelper::decrypt($existing_wallet['skey_encrypted']);
}

?>

<div class="wrap">
    <h1>Policy Wallet Manager</h1>

    <script>
    // Copy functions for clipboard operations
    function copyToClipboard(text) {
        navigator.clipboard.writeText(text).then(function() {
            alert('✅ Address copied to clipboard!');
        }, function(err) {
            console.error('Failed to copy: ', err);
            alert('❌ Failed to copy address. Please copy manually.');
        });
    }

    function copyMnemonicToClipboard() {
        const mnemonicElements = document.querySelectorAll('.mnemonic-word');
        const words = Array.from(mnemonicElements).map(el => el.textContent.trim());
        const mnemonic = words.join(' ');

        navigator.clipboard.writeText(mnemonic).then(function() {
            alert('✅ Mnemonic copied! Store it safely offline.');
        }, function(err) {
            alert('❌ Failed to copy. Please copy manually from above.');
        });
    }

    // Toggle extended key visibility
    let extendedKeyRevealed = false;
    function toggleExtendedKey() {
        const display = document.getElementById('extended-key-display');
        const button = document.getElementById('toggle-extended-key');

        if (!extendedKeyRevealed) {
            display.style.filter = 'none';
            display.style.userSelect = 'text';
            button.textContent = '🙈 Hide Extended Key';
            extendedKeyRevealed = true;
        } else {
            display.style.filter = 'blur(8px)';
            display.style.userSelect = 'none';
            button.textContent = '👁️ Reveal Extended Key (128 chars)';
            extendedKeyRevealed = false;
        }
    }
    </script>

    <?php if ($mnemonic): ?>
        <!-- CRITICAL: Display Mnemonic ONE TIME -->
        <div class="notice notice-error" style="border-left: 4px solid #dc3545; background: #fff;">
            <h2 style="color: #dc3545; margin-top: 10px;">🔴 CRITICAL: SAVE YOUR RECOVERY PHRASE NOW!</h2>
            <p style="font-size: 16px;"><strong>⚠️ THIS WILL ONLY BE SHOWN ONCE! Make sure your screen is private!</strong></p>
            <p>Write down these 24 words in order and store them in a secure location. You will need this to recover your policy wallet if the database is lost.</p>

            <div style="background: #000; color: #0f0; padding: 20px; border-radius: 8px; font-family: 'Courier New', monospace; font-size: 18px; margin: 20px 0; user-select: all;">
                <?php
                $words = explode(' ', $mnemonic);
                $chunks = array_chunk($words, 6);
                foreach ($chunks as $chunk) {
                    echo implode('  ', $chunk) . '<br>';
                }
                ?>
            </div>

            <button type="button" class="button button-primary button-large" onclick="copyMnemonicText('<?php echo esc_js($mnemonic); ?>')">
                📋 Copy to Clipboard
            </button>

            <script>
            function copyMnemonicText(mnemonic) {
                navigator.clipboard.writeText(mnemonic).then(function() {
                    alert('✅ Mnemonic copied! Store it safely offline.');
                }, function(err) {
                    alert('❌ Failed to copy. Please copy manually from above.');
                });
            }
            </script>
        </div>
    <?php endif; ?>

    <div class="notice notice-info">
        <p><strong>What is a Policy Wallet?</strong></p>
        <p>A policy wallet is a secure server-side wallet used to sign minting transactions. This ensures only your server can authorize NFT mints under your policies, preventing unauthorized minting.</p>
        <p><strong>Current Network:</strong> <code><?php echo esc_html(strtoupper($network)); ?></code></p>
    </div>

    <?php if ($existing_wallet): ?>
        <!-- Existing Wallet Display -->
        <div class="card" style="max-width: 800px; margin-top: 20px;">
            <h2>✅ Policy Wallet Active</h2>

            <table class="form-table">
                <tr>
                    <th>Wallet Name</th>
                    <td><?php echo esc_html($existing_wallet['wallet_name']); ?></td>
                </tr>
                <tr>
                    <th>Payment Address</th>
                    <td>
                        <code style="display: block; padding: 10px; background: #f0f0f0; border-radius: 4px; word-break: break-all;">
                            <?php echo esc_html($existing_wallet['payment_address']); ?>
                        </code>
                        <button type="button" class="button button-small" onclick="copyToClipboard('<?php echo esc_js($existing_wallet['payment_address']); ?>')">
                            Copy Address
                        </button>
                    </td>
                </tr>
                <tr>
                    <th>Payment KeyHash</th>
                    <td>
                        <code style="display: block; padding: 10px; background: #f0f0f0; border-radius: 4px; word-break: break-all;">
                            <?php echo esc_html($existing_wallet['payment_keyhash']); ?>
                        </code>
                        <p class="description">This keyhash is used in policy scripts to require your signature for minting.</p>
                    </td>
                </tr>
                <?php if ($extended_key_hex): ?>
                <tr>
                    <th>Extended Signing Key</th>
                    <td>
                        <div style="position: relative;">
                            <code id="extended-key-display" style="display: block; padding: 10px; background: #f0f0f0; border-radius: 4px; word-break: break-all; filter: blur(8px); user-select: none;">
                                <?php echo esc_html($extended_key_hex); ?>
                            </code>
                            <button type="button" id="toggle-extended-key" class="button button-small" style="margin-top: 10px;" onclick="toggleExtendedKey()">
                                👁️ Reveal Extended Key (128 chars)
                            </button>
                        </div>
                        <p class="description">
                            <strong>⚠️ Extended key for CIP-1852 signing.</strong>
                            This is your full 128-character extended signing key (kL||kR).
                            Keep this secret - it can sign transactions for this wallet.
                        </p>
                    </td>
                </tr>
                <?php endif; ?>
                <tr>
                    <th>Network</th>
                    <td><code><?php echo esc_html(strtoupper($existing_wallet['network'])); ?></code></td>
                </tr>
                <tr>
                    <th>Created</th>
                    <td><?php echo esc_html($existing_wallet['created_at']); ?></td>
                </tr>
            </table>

            <div style="margin-top: 20px; padding: 15px; background: #fff3cd; border-left: 4px solid #ffc107;">
                <h3 style="margin-top: 0;">⚠️ Security Notice</h3>
                <p><strong>Your mnemonic phrase and signing keys are encrypted in the database.</strong></p>
                <p>Never share your policy wallet mnemonic or expose it publicly. This wallet authorizes all minting operations on your server.</p>
            </div>

            <div style="margin-top: 20px; display: flex; gap: 10px;">
                <button type="button" class="button button-secondary" onclick="archivePolicyWallet(<?php echo esc_js($existing_wallet['id']); ?>, '<?php echo esc_js($existing_wallet['wallet_name']); ?>')" style="background: #f97316; color: white; border-color: #f97316;">
                    📦 Archive Wallet
                </button>
                <button type="button" class="button button-secondary" onclick="if(confirm('⚠️ Are you sure you want to DELETE this policy wallet?\n\nThis will permanently delete the wallet and you will NOT be able to mint policies created with this wallet.\n\nThis cannot be undone!')) document.getElementById('delete-wallet-form').submit();" style="color: #d63638;">
                    🗑️ Delete Policy Wallet
                </button>
            </div>

            <form id="delete-wallet-form" method="post" style="display: none;">
                <?php wp_nonce_field('cardano_delete_policy_wallet', 'delete_wallet_nonce'); ?>
                <input type="hidden" name="action" value="delete_policy_wallet">
                <input type="hidden" name="wallet_id" value="<?php echo esc_attr($existing_wallet['id']); ?>">
            </form>
        </div>

    <?php else: ?>
        <!-- Generate New Wallet Form -->
        <div class="card" style="max-width: 600px; margin-top: 20px;">
            <h2>🔐 Generate Policy Wallet</h2>
            <p>Create a new secure policy wallet for <strong><?php echo esc_html(strtoupper($network)); ?></strong> network.</p>

            <form method="post" id="generate-wallet-form">
                <?php wp_nonce_field('cardano_generate_policy_wallet', 'generate_wallet_nonce'); ?>
                <input type="hidden" name="action" value="generate_policy_wallet">

                <table class="form-table">
                    <tr>
                        <th><label for="wallet_name">Wallet Name</label></th>
                        <td>
                            <input type="text" id="wallet_name" name="wallet_name" value="<?php echo esc_attr(ucfirst($network) . ' Policy Wallet'); ?>" class="regular-text">
                            <p class="description">A friendly name for this wallet (for your reference only)</p>
                        </td>
                    </tr>
                </table>

                <p class="submit">
                    <button type="submit" class="button button-primary button-large" id="generate-btn">
                        🔑 Generate Secure Policy Wallet
                    </button>
                    <span id="generating-spinner" style="display: none; margin-left: 10px;">
                        <span class="spinner is-active" style="float: none; margin: 0;"></span>
                        Generating wallet... (this may take a few seconds)
                    </span>
                </p>
            </form>

            <div style="margin-top: 20px; padding: 15px; background: #e7f3ff; border-left: 4px solid #2196F3;">
                <h3 style="margin-top: 0;">ℹ️ What happens when you generate a wallet?</h3>
                <ul style="margin-bottom: 0;">
                    <li>A 24-word mnemonic phrase is generated using cryptographic randomness</li>
                    <li>Payment and stake keys are derived using BIP39/CIP-1852 standards</li>
                    <li>The mnemonic and signing key are encrypted with WordPress salts</li>
                    <li>Encrypted data is stored securely in your database</li>
                    <li>The payment keyhash is used to create signature-required policies</li>
                </ul>
            </div>
        </div>

        <script>
        document.getElementById('generate-wallet-form').addEventListener('submit', function(e) {
            console.log('🔑 Policy Wallet Generation - Form submitted');
            console.log('Form action:', this.action);
            console.log('Form method:', this.method);
            console.log('Wallet name:', document.getElementById('wallet_name').value);

            document.getElementById('generate-btn').disabled = true;
            document.getElementById('generating-spinner').style.display = 'inline-block';

            console.log('🔄 Submit button disabled, spinner shown');
            console.log('⏳ Waiting for server response...');
        });

        // Log when page loads/reloads
        console.log('📄 Policy Wallet Manager page loaded');
        console.log('Current URL:', window.location.href);
        console.log('Has "created" param?', window.location.search.includes('created=1'));
        </script>

    <?php endif; ?>

    <!-- Archived Policy Wallets Section -->
    <?php if ($archived_count > 0): ?>
    <div id="archived-wallets" style="margin-top: 30px; padding: 20px; background: #f9fafb; border: 2px solid #e5e7eb; border-radius: 8px; cursor: pointer;" onclick="toggleArchivedWallets()">
        <h2 style="margin: 0 0 15px 0; display: flex; align-items: center; justify-content: space-between;">
            <span>📦 Archived Policy Wallets (<?php echo esc_html($archived_count); ?> / 10 max)</span>
            <span id="archived-wallets-toggle-icon" style="font-size: 18px;">▼</span>
        </h2>
        <p style="margin: 0 0 15px 0; color: #666; font-size: 13px;">
            Archived wallets are preserved but cannot sign mints. Unarchive to restore minting capability.
        </p>

        <div id="archived-wallets-content" style="display: none;">
            <?php foreach ($archived_wallets as $wallet): ?>
                <?php
                // Get policy count for this wallet
                $policy_count = MintModel::countPoliciesByWalletKeyhash($wallet['payment_keyhash'], true);
                ?>
                <div style="background: white; padding: 20px; margin-bottom: 15px; border-radius: 6px; border: 1px solid #ddd;">
                    <h3 style="margin: 0 0 10px 0; color: #374151;">
                        📋 <?php echo esc_html($wallet['wallet_name']); ?>
                    </h3>
                    <div style="display: grid; grid-template-columns: repeat(auto-fit, minmax(250px, 1fr)); gap: 15px; margin-bottom: 15px;">
                        <div>
                            <strong>Payment Address:</strong><br>
                            <span style="font-family: monospace; font-size: 11px; color: #666; word-break: break-all;">
                                <?php echo esc_html(substr($wallet['payment_address'], 0, 30)); ?>...
                            </span>
                        </div>
                        <div>
                            <strong>KeyHash:</strong><br>
                            <span style="font-family: monospace; font-size: 11px; color: #666;">
                                <?php echo esc_html(substr($wallet['payment_keyhash'], 0, 20)); ?>...
                            </span>
                        </div>
                        <div>
                            <strong>Policies:</strong> <?php echo esc_html($policy_count); ?>
                        </div>
                        <div>
                            <strong>Archived:</strong> <?php echo esc_html($wallet['archived_at'] ? date('M j, Y', strtotime($wallet['archived_at'])) : 'N/A'); ?>
                        </div>
                    </div>

                    <div style="display: flex; gap: 10px;">
                        <button type="button" class="button button-primary" onclick="event.stopPropagation(); unarchivePolicyWallet(<?php echo esc_js($wallet['id']); ?>, '<?php echo esc_js($wallet['wallet_name']); ?>')">
                            ↑ Unarchive
                        </button>
                        <button type="button" class="button button-secondary" onclick="event.stopPropagation(); toggleWalletDetails('wallet-<?php echo esc_attr($wallet['id']); ?>')">
                            👁 View Details
                        </button>
                        <button type="button" class="button button-secondary" onclick="event.stopPropagation(); if(confirm('⚠️ Permanently delete this archived wallet?\n\nThis will delete the wallet but NOT the policies. You will not be able to mint those policies anymore.\n\nThis cannot be undone!')) deleteArchivedWallet(<?php echo esc_js($wallet['id']); ?>);" style="color: #d63638;">
                            🗑 Delete Permanently
                        </button>
                    </div>

                    <div id="wallet-<?php echo esc_attr($wallet['id']); ?>" style="display: none; margin-top: 15px; padding-top: 15px; border-top: 1px solid #e5e7eb;">
                        <h4 style="margin: 0 0 10px 0;">Full Wallet Details</h4>
                        <table style="width: 100%; font-size: 13px;">
                            <tr>
                                <th style="padding: 8px; text-align: left; background: #f3f4f6;">Payment Address</th>
                                <td style="padding: 8px; font-family: monospace; font-size: 11px; word-break: break-all;"><?php echo esc_html($wallet['payment_address']); ?></td>
                            </tr>
                            <tr>
                                <th style="padding: 8px; text-align: left; background: #f3f4f6;">Payment KeyHash</th>
                                <td style="padding: 8px; font-family: monospace; font-size: 11px; word-break: break-all;"><?php echo esc_html($wallet['payment_keyhash']); ?></td>
                            </tr>
                            <tr>
                                <th style="padding: 8px; text-align: left; background: #f3f4f6;">Stake Address</th>
                                <td style="padding: 8px; font-family: monospace; font-size: 11px; word-break: break-all;"><?php echo esc_html($wallet['stake_address'] ?: 'N/A'); ?></td>
                            </tr>
                            <tr>
                                <th style="padding: 8px; text-align: left; background: #f3f4f6;">Created</th>
                                <td style="padding: 8px;"><?php echo esc_html($wallet['created_at']); ?></td>
                            </tr>
                            <tr>
                                <th style="padding: 8px; text-align: left; background: #f3f4f6;">Archived</th>
                                <td style="padding: 8px;"><?php echo esc_html($wallet['archived_at'] ?: 'N/A'); ?></td>
                            </tr>
                        </table>
                    </div>
                </div>
            <?php endforeach; ?>
        </div>
    </div>
    <?php endif; ?>

</div>

<script>
// Archive wallet
function archivePolicyWallet(walletId, walletName) {
    if (!confirm('Archive "' + walletName + '"?\n\nThis will make ALL policies created with this wallet read-only (unmintable) until you unarchive it.\n\nThe wallet and policies will be preserved.')) {
        return;
    }

    const formData = new FormData();
    formData.append('action', 'cardano_archive_policy_wallet');
    formData.append('nonce', '<?php echo wp_create_nonce('cardanocheckoutnonce'); ?>');
    formData.append('wallet_id', walletId);

    fetch(ajaxurl, {
        method: 'POST',
        body: formData
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            alert('✅ Policy wallet archived successfully!');
            location.reload();
        } else {
            alert('❌ Error: ' + (data.data?.message || 'Failed to archive wallet'));
        }
    })
    .catch(error => {
        console.error('Archive error:', error);
        alert('❌ Network error while archiving wallet');
    });
}

// Unarchive wallet
function unarchivePolicyWallet(walletId, walletName) {
    if (!confirm('Restore "' + walletName + '" to active?\n\nThis will make it the active signing wallet for mints.')) {
        return;
    }

    const formData = new FormData();
    formData.append('action', 'cardano_unarchive_policy_wallet');
    formData.append('nonce', '<?php echo wp_create_nonce('cardanocheckoutnonce'); ?>');
    formData.append('wallet_id', walletId);

    fetch(ajaxurl, {
        method: 'POST',
        body: formData
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            alert('✅ Policy wallet unarchived successfully!');
            location.reload();
        } else {
            alert('❌ Error: ' + (data.data?.message || 'Failed to unarchive wallet'));
        }
    })
    .catch(error => {
        console.error('Unarchive error:', error);
        alert('❌ Network error while unarchiving wallet');
    });
}

// Delete archived wallet
function deleteArchivedWallet(walletId) {
    const formData = new FormData();
    formData.append('action', 'delete_policy_wallet');
    formData.append('nonce', '<?php echo wp_create_nonce('cardano_delete_policy_wallet'); ?>');
    formData.append('wallet_id', walletId);

    fetch(ajaxurl, {
        method: 'POST',
        body: formData
    })
    .then(response => response.json())
    .then(data => {
        alert('✅ Wallet deleted successfully!');
        location.reload();
    })
    .catch(error => {
        console.error('Delete error:', error);
        alert('❌ Network error while deleting wallet');
    });
}

// Toggle archived wallets section
function toggleArchivedWallets() {
    const content = document.getElementById('archived-wallets-content');
    const icon = document.getElementById('archived-wallets-toggle-icon');

    if (content.style.display === 'none') {
        content.style.display = 'block';
        icon.textContent = '▲';
    } else {
        content.style.display = 'none';
        icon.textContent = '▼';
    }
}

// Toggle wallet details
function toggleWalletDetails(elementId) {
    const details = document.getElementById(elementId);
    const button = event.target;

    if (details.style.display === 'none') {
        details.style.display = 'block';
        button.textContent = '👁 Hide Details';
    } else {
        details.style.display = 'none';
        button.textContent = '👁 View Details';
    }
}
</script>

<style>
.card h2 {
    margin-top: 0;
    padding-bottom: 10px;
    border-bottom: 1px solid #ddd;
}
</style>
