<?php
namespace CardanoMintPay\Models;

if ( ! defined( 'ABSPATH' ) ) exit;

class OrderModel {
    private static function table() {
        global $wpdb;
        return $wpdb->prefix . 'cardanopay_orders';
    }

    public static function insert($args) {
        global $wpdb;
        $data = [
            'orderId' => isset($args['order_id']) ? sanitize_text_field($args['order_id']) : '',
            'orderDate' => isset($args['order_date']) ? sanitize_text_field($args['order_date']) : self::getNewYorkTime(),
            'firstName' => isset($args['first_name']) ? sanitize_text_field($args['first_name']) : '',
            'lastName' => isset($args['last_name']) ? sanitize_text_field($args['last_name']) : '',
            'email' => isset($args['email']) ? sanitize_email($args['email']) : '',
            'phone' => isset($args['phone']) ? sanitize_text_field($args['phone']) : '',
            'shippingAddress1' => isset($args['shipping_address_1']) ? sanitize_text_field($args['shipping_address_1']) : '',
            'shippingAddress2' => isset($args['shipping_address_2']) ? sanitize_text_field($args['shipping_address_2']) : '',
            'city' => isset($args['city']) ? sanitize_text_field($args['city']) : '',
            'stateProvince' => isset($args['state_province']) ? sanitize_text_field($args['state_province']) : '',
            'postalCode' => isset($args['postal_code']) ? sanitize_text_field($args['postal_code']) : '',
            'country' => isset($args['country']) ? sanitize_text_field($args['country']) : '',
            'specialInstructions' => isset($args['special_instructions']) ? sanitize_textarea_field($args['special_instructions']) : '',
            'productId' => isset($args['product_id']) ? sanitize_text_field($args['product_id']) : '',
            'productName' => isset($args['product_name']) ? sanitize_text_field($args['product_name']) : '',
            'quantity' => isset($args['quantity']) ? intval($args['quantity']) : 1,
            'orderTotalUsd' => isset($args['order_total_usd']) ? floatval($args['order_total_usd']) : 0,
            'orderTotalAda' => isset($args['order_total_ada']) ? floatval($args['order_total_ada']) : 0,
            'status' => isset($args['status']) ? sanitize_text_field($args['status']) : 'pending',
            'txHash' => isset($args['tx_hash']) ? sanitize_text_field($args['tx_hash']) : '',
            'createdAt' => self::getNewYorkTime()
        ];
        $wpdb->insert(self::table(), $data);
        return $wpdb->insert_id;
    }

    public static function getByOrderId($order_id) {
        global $wpdb;
        return $wpdb->get_row(
            $wpdb->prepare(
                "SELECT * FROM " . self::table() . " WHERE orderId = %s",
                $order_id
            ),
            ARRAY_A
        );
    }

    public static function listOrders($limit = 50, $offset = 0, $orderby = 'createdAt', $order = 'DESC') {
        global $wpdb;
        $allowed_orderby = ['orderId', 'orderDate', 'productName', 'country', 'createdAt', 'lastName'];
        $allowed_order = ['ASC', 'DESC'];
        
        $orderby = in_array($orderby, $allowed_orderby) ? $orderby : 'createdAt';
        $order = in_array(strtoupper($order), $allowed_order) ? strtoupper($order) : 'DESC';
        
        return $wpdb->get_results(
            $wpdb->prepare(
                "SELECT * FROM " . self::table() . " ORDER BY $orderby $order LIMIT %d OFFSET %d",
                $limit, $offset
            ),
            ARRAY_A
        );
    }

    public static function countOrders() {
        global $wpdb;
        return $wpdb->get_var("SELECT COUNT(*) FROM " . self::table());
    }

    public static function updateStatus($order_id, $status) {
        global $wpdb;
        return $wpdb->update(
            self::table(),
            ['status' => sanitize_text_field($status)],
            ['orderId' => sanitize_text_field($order_id)]
        );
    }

    public static function deleteOrder($order_id) {
        global $wpdb;
        return $wpdb->delete(
            self::table(),
            ['orderId' => sanitize_text_field($order_id)]
        );
    }

    public static function deleteAllOrders() {
        global $wpdb;
        return $wpdb->query("DELETE FROM " . self::table());
    }

    public static function exportOrders() {
        global $wpdb;
        return $wpdb->get_results(
            "SELECT * FROM " . self::table() . " ORDER BY createdAt DESC",
            ARRAY_A
        );
    }

    public static function installTable() {
        global $wpdb;
        $table = self::table();
        $charset_collate = $wpdb->get_charset_collate();
        $sql = "CREATE TABLE IF NOT EXISTS $table (
            id bigint(20) unsigned NOT NULL AUTO_INCREMENT,
            orderId varchar(50) NOT NULL,
            orderDate datetime NOT NULL,
            firstName varchar(100) NOT NULL,
            lastName varchar(100) NOT NULL,
            email varchar(255) NOT NULL,
            phone varchar(50) NOT NULL,
            shippingAddress1 varchar(255) NOT NULL,
            shippingAddress2 varchar(255),
            city varchar(100) NOT NULL,
            stateProvince varchar(100) NOT NULL,
            postalCode varchar(20) NOT NULL,
            country varchar(100) NOT NULL,
            specialInstructions text,
            productId varchar(50) NOT NULL,
            productName varchar(255) NOT NULL,
            quantity int(11) NOT NULL DEFAULT 1,
            orderTotalUsd decimal(10,2) NOT NULL DEFAULT 0,
            orderTotalAda decimal(18,6) NOT NULL DEFAULT 0,
            status varchar(20) NOT NULL DEFAULT 'pending',
            txHash varchar(100),
            createdAt datetime NOT NULL,
            PRIMARY KEY  (id),
            UNIQUE KEY orderId (orderId),
            KEY orderDate (orderDate),
            KEY productName (productName),
            KEY country (country)
        ) $charset_collate;";

        require_once(ABSPATH . 'wp-admin/includes/upgrade.php');
        dbDelta($sql);
        
        // Add txHash column if it doesn't exist (for existing installations)
        $column_exists = $wpdb->get_results("SHOW COLUMNS FROM $table LIKE 'txHash'");
        if (empty($column_exists)) {
            $wpdb->query("ALTER TABLE $table ADD COLUMN txHash varchar(100) AFTER status");
        }
    }
    
    /**
     * Get current time in New York timezone
     */
    private static function getNewYorkTime() {
        // Force UTC first, then convert to New York
        $utc = new \DateTimeZone('UTC');
        $ny = new \DateTimeZone('America/New_York');
        
        $datetime = new \DateTime('now', $utc);
        $datetime->setTimezone($ny);
        
        return $datetime->format('Y-m-d H:i:s');
    }
}
