<?php
namespace CardanoMintPay\Helpers;

class AnvilAPI_Pay {
    
    /**
     * Call Anvil API endpoint
     */
    public static function call($endpoint, $data, $plugin_type = 'pay') {
        $api_url = get_option("cardano_{$plugin_type}_anvil_api_url", 'https://preprod.api.ada-anvil.app/v2/services');
        $api_key = get_option("cardano_{$plugin_type}_anvil_api_key");
        
        if (!$api_key) {
            return new \WP_Error('no_api_key', 'Anvil API key not configured');
        }
        
        $response = wp_remote_post($api_url . '/' . $endpoint, array(
            'headers' => array(
                'Content-Type' => 'application/json',
                'X-Api-Key' => $api_key
            ),
            'body' => wp_json_encode($data),
            'timeout' => 30
        ));
        
        if (is_wp_error($response)) {
            return $response;
        }
        
        $body = wp_remote_retrieve_body($response);
        $decoded = json_decode($body, true);
        
        if (wp_remote_retrieve_response_code($response) !== 200) {
            return new \WP_Error('api_error', $decoded['message'] ?? 'API request failed');
        }
        
        return $decoded;
    }
    
    /**
     * Build transaction for payment
     */
    public static function buildPaymentTransaction($merchant_address, $customer_address, $usd_price, $product_name, $plugin_type = 'pay') {
        // Get current ADA price
        $ada_price = self::getAdaPrice();
        $ada_amount = $usd_price / $ada_price;
        $receipt_amount = 1.0; // 1 ADA receipt back to customer
        
        // Build transaction request
        $transaction_request = array(
            'changeAddress' => $customer_address,
            'outputs' => array(
                array(
                    'address' => $merchant_address,
                    'lovelace' => intval($ada_amount * 1000000)
                ),
                array(
                    'address' => $customer_address, 
                    'lovelace' => intval($receipt_amount * 1000000)
                )
            ),
            'message' => array(
                'Purchase Receipt - ' . get_bloginfo('name'),
                'Product: ' . $product_name,
                'Price: $' . number_format($usd_price, 2) . ' USD (' . number_format($ada_amount, 2) . ' ADA)',
                'Receipt: 1 ADA returned to customer',
                'Timestamp: ' . current_time('c'),
                'Exchange Rate: $' . number_format($ada_price, 4) . '/ADA'
            )
        );
        
        return self::call('transactions/build', $transaction_request, $plugin_type);
    }
    
    /**
     * Submit transaction to blockchain
     */
    public static function submitTransaction($transaction, $signatures, $plugin_type = 'pay') {
        $submit_data = array(
            'transaction' => $transaction,
            'signatures' => is_array($signatures) ? $signatures : array()
        );
        
        return self::call('transactions/submit', $submit_data, $plugin_type);
    }
    
    /**
     * Get ADA price with caching (shared between both plugins)
     */
    public static function getAdaPrice() {
        $cached_price = get_transient('cardano_anvil_ada_price');
        
        if ($cached_price !== false && $cached_price > 0) {
            return floatval($cached_price);
        }
        
        // Fetch from CoinGecko API
        $response = wp_remote_get('https://api.coingecko.com/api/v3/simple/price?ids=cardano&vs_currencies=usd', array(
            'timeout' => 10,
            'user-agent' => 'WordPress/Cardano-MintPay-Plugin'
        ));
        
        if (!is_wp_error($response)) {
            $body = wp_remote_retrieve_body($response);
            $data = json_decode($body, true);
            
            if (isset($data['cardano']['usd']) && $data['cardano']['usd'] > 0) {
                $price = floatval($data['cardano']['usd']);
                set_transient('cardano_anvil_ada_price', $price, 300); // 5 minutes
                return $price;
            }
        }
        
        return 1.0; // Fallback
    }
}

