<?php
namespace CardanoMintPay\Helpers;

class AnvilAPI_Mint {
    
    /**
     * Call Anvil API endpoint
     */
    public static function call($endpoint, $data, $plugin_type = 'mint') {
        $api_url = get_option("cardano_{$plugin_type}_anvil_api_url", 'https://preprod.api.ada-anvil.app/v2/services');
        $api_key = get_option("cardano_{$plugin_type}_anvil_api_key");
        
        error_log("Anvil API call details:");
        error_log("Endpoint: " . $endpoint);
        error_log("API URL: " . $api_url);
        error_log("Plugin type: " . $plugin_type);
        error_log("Data being sent: " . wp_json_encode($data, JSON_PRETTY_PRINT));
        
        if (!$api_key) {
            error_log("No API key configured for plugin type: " . $plugin_type);
            return new \WP_Error('no_api_key', 'Anvil API key not configured');
        }
        
        error_log("Making API call to: " . $api_url . '/' . $endpoint);
        $response = wp_remote_post($api_url . '/' . $endpoint, array(
            'headers' => array(
                'Content-Type' => 'application/json',
                'X-Api-Key' => $api_key
            ),
            'body' => wp_json_encode($data),
            'timeout' => 30
        ));
        
        if (is_wp_error($response)) {
            return $response;
        }
        
        $body = wp_remote_retrieve_body($response);
        $decoded = json_decode($body, true);
        
        if (wp_remote_retrieve_response_code($response) !== 200) {
            return new \WP_Error('api_error', $decoded['message'] ?? 'API request failed');
        }
        
        return $decoded;
    }
    
    /**
     * Build transaction for payment
     */
    public static function buildPaymentTransaction($merchant_address, $customer_address, $usd_price, $product_name, $plugin_type = 'mint') {
        // Convert addresses to Bech32 format if needed
        $merchant_address = self::convertAddressToBech32($merchant_address, $plugin_type);
        $customer_address = self::convertAddressToBech32($customer_address, $plugin_type);
        
        // Get current ADA price
        $ada_price = self::getAdaPrice();
        $ada_amount = $usd_price / $ada_price;
        $receipt_amount = 1.0; // 1 ADA receipt back to customer
        
        // Build transaction request
        $transaction_request = array(
            'changeAddress' => $customer_address,
            'outputs' => array(
                array(
                    'address' => $merchant_address,
                    'lovelace' => intval($ada_amount * 1000000)
                ),
                array(
                    'address' => $customer_address, 
                    'lovelace' => intval($receipt_amount * 1000000)
                )
            ),
            'message' => array(
                self::truncateMetadata('Purchase Receipt - ' . get_bloginfo('name')),
                self::truncateMetadata('Product: ' . $product_name),
                self::truncateMetadata('Price: $' . number_format($usd_price, 2) . ' USD (' . number_format($ada_amount, 2) . ' ADA)'),
                self::truncateMetadata('Receipt: 1 ADA returned to customer'),
                self::truncateMetadata('Timestamp: ' . current_time('c')),
                self::truncateMetadata('Exchange Rate: $' . number_format($ada_price, 4) . '/ADA')
            )
        );
        
        return self::call('transactions/build', $transaction_request, $plugin_type);
    }
    
    /**
     * Truncate metadata string to fit Cardano's 64-character limit
     */
    public static function truncateMetadata($string) {
        $max_length = 64;
        if (strlen($string) <= $max_length) {
            return $string;
        }
        
        // Truncate and add ellipsis
        return substr($string, 0, $max_length - 3) . '...';
    }

    /**
     * Convert CBOR-encoded address to Bech32 format using Anvil API
     */
    public static function convertAddressToBech32($address, $plugin_type = 'mint') {
        // Check if it's already in Bech32 format
        if (preg_match('/^addr[0-9a-z]+$/', $address)) {
            return $address;
        }
        
        // Try to convert using Anvil API
        $response = self::call('utils/addresses/parse', array('address' => $address), $plugin_type);
        
        if (is_wp_error($response)) {
            // If conversion fails, return original address
            return $address;
        }
        
        // Extract Bech32 address from response
        error_log("Checking response format for address conversion:");
        error_log("response['address'] exists: " . (isset($response['address']) ? 'YES' : 'NO'));
        error_log("response['bech32Address'] exists: " . (isset($response['bech32Address']) ? 'YES' : 'NO'));
        error_log("response['parsed']['address'] exists: " . (isset($response['parsed']['address']) ? 'YES' : 'NO'));
        error_log("response['payment'] exists: " . (isset($response['payment']) ? 'YES' : 'NO'));
        error_log("response['stake'] exists: " . (isset($response['stake']) ? 'YES' : 'NO'));
        
        if (isset($response['address'])) {
            error_log("Using response['address']: " . $response['address']);
            return $response['address'];
        } elseif (isset($response['bech32Address'])) {
            error_log("Using response['bech32Address']: " . $response['bech32Address']);
            return $response['bech32Address'];
        } elseif (isset($response['parsed']['address'])) {
            error_log("Using response['parsed']['address']: " . $response['parsed']['address']);
            return $response['parsed']['address'];
        } elseif (isset($response['payment']) && isset($response['stake'])) {
            // Handle the payment/stake format by constructing the full address
            // Bech32 addresses need the '1' separator between prefix and data
            $fullAddress = 'addr1' . '1' . $response['payment'] . $response['stake'];
            error_log("Payment/stake format detected, constructing Bech32 address: " . $fullAddress);
            return $fullAddress;
        }
        
        error_log("No Bech32 address found in response, returning original address: " . $address);
        return $address;
    }

    /**
     * Build transaction for NFT minting with CIP-25 metadata
     */
    public static function buildMintTransaction($merchant_address, $customer_address, $usd_price, $policy_id, $plugin_type = 'mint', $mint_data = null) {
        error_log("buildMintTransaction called with:");
        error_log("merchant_address: " . $merchant_address);
        error_log("customer_address: " . $customer_address);
        error_log("usd_price: " . $usd_price);
        error_log("policy_id: " . $policy_id);
        error_log("plugin_type: " . $plugin_type);

        // CRITICAL: Convert all addresses to Bech32 format for consistency
        error_log("Converting addresses to Bech32 format for consistency...");

        // Convert merchant address to Bech32 if needed
        if (!preg_match('/^addr[0-9a-z]+$/', $merchant_address)) {
            $merchant_address = self::convertAddressToBech32($merchant_address, $plugin_type);
            error_log("Converted merchant address to Bech32: " . $merchant_address);
        }

        // Convert customer address to Bech32 if needed
        if (!preg_match('/^addr[0-9a-z]+$/', $customer_address)) {
            $customer_address = self::convertAddressToBech32($customer_address, $plugin_type);
            error_log("Converted customer address to Bech32: " . $customer_address);
        }

        error_log("Final addresses - Merchant: " . $merchant_address . ", Customer: " . $customer_address);

        // Get current ADA price
        error_log("Getting ADA price...");
        $ada_price = self::getAdaPrice();
        error_log("ADA price: " . $ada_price);
        $ada_amount = $usd_price / $ada_price;
        error_log("ADA amount: " . $ada_amount);

        $receipt_amount = 2.0; // 2 ADA minimum for NFT output
        error_log("Receipt amount: " . $receipt_amount);

        // Generate unique asset name for this mint
        // Anvil API will handle encoding when we specify format: "utf8"
        $asset_name_raw = 'NFT_' . time() . '_' . substr(md5($customer_address . $policy_id), 0, 8);
        error_log("Asset name: " . $asset_name_raw);

        // Get mint-specific metadata if available
        $nft_name = $asset_name_raw;
        $nft_image = '';
        $nft_description = 'NFT minted via ' . get_bloginfo('name');
        $nft_metadata_attributes = array();  // Additional attributes from metadata builder
        $policy_script = null;

        if ($mint_data) {
            // Use mint title as default NFT name
            $nft_name = $mint_data['title'] ?? $asset_name_raw;

            // Debug: Log what we have in mint_data
            error_log("=== IMAGE SOURCE DEBUG ===");
            error_log("ipfs_cid_manual: " . ($mint_data['ipfs_cid_manual'] ?? 'NOT SET'));
            error_log("ipfs_cid: " . ($mint_data['ipfs_cid'] ?? 'NOT SET'));
            error_log("image_id: " . ($mint_data['image_id'] ?? 'NOT SET'));

            // Get image URL - priority: Manual IPFS > Pinata IPFS > WordPress CDN
            if (isset($mint_data['ipfs_cid_manual']) && !empty($mint_data['ipfs_cid_manual'])) {
                // Priority 1: Manually pasted IPFS hash (user's own pinning service)
                $nft_image = 'ipfs://' . $mint_data['ipfs_cid_manual'];
                error_log("Using manually pasted IPFS image: " . $nft_image);
            } elseif (isset($mint_data['ipfs_cid']) && !empty($mint_data['ipfs_cid'])) {
                // Priority 2: Pinata IPFS CID (CIDv0 format: Qm...)
                $nft_image = 'ipfs://' . $mint_data['ipfs_cid'];
                error_log("Using Pinata IPFS image: " . $nft_image);
            } elseif (isset($mint_data['image_id']) && $mint_data['image_id']) {
                // Priority 3: WordPress CDN URL (fallback)
                $image_url = wp_get_attachment_url($mint_data['image_id']);
                if ($image_url) {
                    $nft_image = $image_url;
                    error_log("Using WordPress CDN image: " . $nft_image);
                }
            }

            // Parse NFT metadata from metadata builder
            if (isset($mint_data['nft_metadata']) && !empty($mint_data['nft_metadata'])) {
                $user_metadata = json_decode($mint_data['nft_metadata'], true);
                if (is_array($user_metadata)) {
                    error_log("User metadata from builder: " . wp_json_encode($user_metadata, JSON_PRETTY_PRINT));

                    // Override with user-defined values if present
                    if (isset($user_metadata['name'])) {
                        $nft_name = $user_metadata['name'];
                    }
                    if (isset($user_metadata['description'])) {
                        $nft_description = $user_metadata['description'];
                    }

                    // Handle image field from metadata builder
                    // Only use metadata builder image if:
                    // 1. No manual IPFS hash
                    // 2. No Pinata IPFS hash
                    // 3. No WordPress media library image
                    // 4. Metadata image is not empty or just "ipfs://"
                    if (isset($user_metadata['image']) && empty($nft_image)) {
                        $metadata_image = trim($user_metadata['image']);
                        // Don't use if it's just the placeholder "ipfs://" or empty
                        if (!empty($metadata_image) && $metadata_image !== 'ipfs://') {
                            $nft_image = $metadata_image;
                        }
                    }

                    // Collect any additional attributes (not name, description, image, mediaType)
                    $reserved_keys = array('name', 'description', 'image', 'mediaType', 'files');
                    foreach ($user_metadata as $key => $value) {
                        if (!in_array($key, $reserved_keys)) {
                            $nft_metadata_attributes[$key] = $value;
                        }
                    }
                }
            }

            // Get policy script schema if available
            // The policy_json field contains the full policy generation response
            // We need to extract the 'schema' field for preloadedScripts
            if (isset($mint_data['policy_json'])) {
                $policy_data = json_decode($mint_data['policy_json'], true);
                if (isset($policy_data['schema'])) {
                    $policy_script = $policy_data['schema'];
                    error_log("Extracted policy schema from policy_json");
                } else {
                    error_log("WARNING: policy_json exists but no 'schema' field found!");
                    error_log("policy_json content: " . wp_json_encode($policy_data, JSON_PRETTY_PRINT));
                }
            }
        }

        // Build CIP-25 metadata for the mint object
        // Anvil API expects metadata fields directly in the mint array with version: "cip25"
        $cip25_metadata = array(
            'name' => $nft_name,
            'image' => $nft_image,
            'description' => $nft_description,
            'mediaType' => 'image/png'
        );

        // Add any additional user-defined metadata attributes
        // IMPORTANT: Don't let empty metadata attributes overwrite our core fields
        foreach ($nft_metadata_attributes as $key => $value) {
            // Skip if trying to override a core field with an empty value
            if (isset($cip25_metadata[$key]) && empty($value)) {
                continue; // Don't overwrite existing value with empty
            }
            $cip25_metadata[$key] = $value;
        }

        // Debug: Final metadata being sent
        error_log("=== FINAL CIP-25 METADATA ===");
        error_log(wp_json_encode($cip25_metadata, JSON_PRETTY_PRINT));

        // Add files array for better explorer compatibility
        if (!empty($nft_image)) {
            $cip25_metadata['files'] = array(
                array(
                    'name' => $nft_name,
                    'mediaType' => 'image/png',
                    'src' => $nft_image
                )
            );
        }

        error_log("CIP-25 metadata: " . wp_json_encode($cip25_metadata, JSON_PRETTY_PRINT));

        // Build transaction request with NFT minting using Anvil API format
        $transaction_request = array(
            'changeAddress' => $customer_address,
            'outputs' => array(
                array(
                    'address' => $merchant_address,
                    'lovelace' => intval($ada_amount * 1000000)
                ),
                array(
                    'address' => $customer_address,
                    'lovelace' => intval($receipt_amount * 1000000),
                    'assets' => array(
                        array(
                            'policyId' => $policy_id,
                            'assetName' => array(
                                'name' => $asset_name_raw,
                                'format' => 'utf8'
                            ),
                            'quantity' => 1
                        )
                    )
                )
            ),
            'mint' => array(
                array(
                    'version' => 'cip25',  // CRITICAL: tells Anvil to generate 721 metadata
                    'policyId' => $policy_id,
                    'quantity' => 1,
                    'assetName' => array(
                        'name' => $asset_name_raw,
                        'format' => 'utf8'
                    ),
                    'metadata' => $cip25_metadata
                )
            )
        );

        // Add preloaded scripts if policy script is available (REQUIRED for minting)
        if ($policy_script) {
            $transaction_request['preloadedScripts'] = array(
                array(
                    'type' => 'simple',
                    'script' => $policy_script,  // JSON schema, not CBOR hex
                    'hash' => $policy_id
                )
            );
            error_log("Added preloaded script to transaction request");
        } else {
            error_log("WARNING: No policy script provided - minting will likely fail!");
        }

        error_log("Transaction request built successfully");
        error_log("Full transaction request: " . wp_json_encode($transaction_request, JSON_PRETTY_PRINT));
        error_log("Calling Anvil API to build transaction...");
        $result = self::call('transactions/build', $transaction_request, $plugin_type);
        error_log("Anvil API call result: " . print_r($result, true));
        return $result;
    }
    
    /**
     * Submit transaction to blockchain with automatic retry for UTXO value mismatches
     */
    public static function submitTransaction($transaction, $signatures, $plugin_type = 'mint') {
        $submit_data = array(
            'transaction' => $transaction,
            'signatures' => is_array($signatures) ? $signatures : array()
        );
        
        // First submission attempt
        $response = self::call('transactions/submit', $submit_data, $plugin_type);
        
        // Check if submission failed due to ValueNotConservedUTxO error
        if (is_wp_error($response)) {
            $error_message = $response->get_error_message();
            
            // Check for UTXO value mismatch error
            if (strpos($error_message, 'ValueNotConservedUTxO') !== false || 
                strpos($error_message, 'Mismatch') !== false) {
                
                error_log("UTXO value mismatch detected, attempting automatic retry...");
                
                // Extract transaction hash for retry
                $tx_hash = null;
                if (is_array($transaction) && isset($transaction['hash'])) {
                    $tx_hash = $transaction['hash'];
                } elseif (is_string($transaction)) {
                    // For string transactions, we'll still attempt retry
                    // The transaction hash will be logged for debugging
                    $tx_hash = 'string_transaction';
                }
                
                if ($tx_hash) {
                    error_log("Retrying transaction with hash: " . $tx_hash);
                    
                    // Wait a brief moment for blockchain state to stabilize
                    sleep(1);
                    
                    // Retry the same transaction (UTXO state may have stabilized)
                    $retry_response = self::call('transactions/submit', $submit_data, $plugin_type);
                    
                    if (!is_wp_error($retry_response)) {
                        error_log("Automatic retry successful!");
                        return $retry_response;
                    } else {
                        error_log("Automatic retry also failed: " . $retry_response->get_error_message());
                    }
                } else {
                    error_log("Could not extract transaction hash for retry");
                }
            }
        }
        
        return $response;
    }
    
    /**
     * Get ADA price with caching (for MINT NOW plugin)
     */
    public static function getAdaPrice() {
        $cached_price = get_transient('cardano_anvil_ada_price');
        
        if ($cached_price !== false && $cached_price > 0) {
            return floatval($cached_price);
        }
        
        // Fetch from CoinGecko API
        $response = wp_remote_get('https://api.coingecko.com/api/v3/simple/price?ids=cardano&vs_currencies=usd', array(
            'timeout' => 10,
            'user-agent' => 'WordPress/Cardano-MintPay-Plugin'
        ));
        
        if (!is_wp_error($response)) {
            $body = wp_remote_retrieve_body($response);
            $data = json_decode($body, true);
            
            if (isset($data['cardano']['usd']) && $data['cardano']['usd'] > 0) {
                $price = floatval($data['cardano']['usd']);
                set_transient('cardano_anvil_ada_price', $price, 300); // 5 minutes
                return $price;
            }
        }
        
        return 1.0; // Fallback
    }
}