<?php
namespace CardanoMintPay\Controllers;
use CardanoMintPay\Models\ProductModel;

class CheckoutController {
    public static function register() {
        add_shortcode('cardano-pay', [self::class, 'renderCheckout']);
        add_action('wp_ajax_cardanobuildtransaction', [self::class, 'ajaxBuildTx']);
        add_action('wp_ajax_nopriv_cardanobuildtransaction', [self::class, 'ajaxBuildTx']);
        add_action('wp_ajax_cardanosubmittransaction', [self::class, 'ajaxSubmitTx']);
        add_action('wp_ajax_nopriv_cardanosubmittransaction', [self::class, 'ajaxSubmitTx']);
        add_action('wp_ajax_cardanogetadaprice', [self::class, 'ajaxGetPrice']);
        add_action('wp_ajax_nopriv_cardanogetadaprice', [self::class, 'ajaxGetPrice']);
    }

    public static function renderCheckout($atts) {
        $atts = shortcode_atts([
            'product-id' => '',
            'productid' => '',
            'productname' => 'Physical Product',
            'price' => 0.00,
            'class' => ''
        ], $atts);

        // Handle both product-id and productid for backward compatibility
        if (!empty($atts['product-id']) && empty($atts['productid'])) {
            $atts['productid'] = $atts['product-id'];
        }

        $product = ProductModel::getProductById($atts['productid']);
        $prod = $product; // Make product available as $prod for the view
        
        // Get merchant address from plugin settings (not from shortcode)
        $merchant_address = get_option('cardano_pay_merchant_address', '');
        
        // Debug: Check if product was found
        if (WP_DEBUG && WP_DEBUG_LOG) {
            error_log('Cardano Pay Controller Debug - Product ID: ' . $atts['productid']);
            error_log('Cardano Pay Controller Debug - Product Found: ' . ($product ? 'YES' : 'NO'));
            error_log('Cardano Pay Controller Debug - Merchant Address: ' . $merchant_address);
            if ($product) {
                error_log('Cardano Pay Controller Debug - Product Name: ' . $product->ProductName);
                error_log('Cardano Pay Controller Debug - Product Price: ' . $product->ProductPrice);
            }
        }
        // Pass merchant address to the view
        $atts['merchantaddress'] = $merchant_address;
        
        ob_start();
        include(plugin_dir_path(__FILE__) . '/../views/checkout-form.php');
        $output = ob_get_clean();
        
        // Apply custom class if provided
        $wrapper_class = 'cardano-shortcode-wrapper';
        if (!empty($atts['class'])) {
            $wrapper_class .= ' ' . sanitize_html_class($atts['class']);
        }
        
        // Prevent WordPress from adding auto-paragraphs to our shortcode output
        return '<div class="' . esc_attr($wrapper_class) . '">' . $output . '</div>';
    }

    public static function ajaxBuildTx() {
        check_ajax_referer('cardanocheckoutnonce', 'nonce');
        
        // Debug logging - log all received POST data (only in development)
        if (WP_DEBUG && WP_DEBUG_LOG) {
            error_log('=== AJAX BuildTx Debug ===');
            error_log('POST Data: ' . print_r($_POST, true));
            error_log('Merchant Address: ' . ($_POST['merchantaddress'] ?? 'NOT SET'));
            error_log('Customer Address: ' . ($_POST['customeraddress'] ?? 'NOT SET'));
            error_log('USD Price: ' . ($_POST['usdprice'] ?? 'NOT SET'));
            error_log('Product Name: ' . ($_POST['productname'] ?? 'NOT SET'));
            error_log('========================');
        }
        
        $merchant = sanitize_text_field($_POST['merchantaddress'] ?? '');
        $customer = sanitize_text_field($_POST['customeraddress'] ?? '');
        $usd = floatval($_POST['usdprice'] ?? 0);
        $prod = sanitize_text_field($_POST['productname'] ?? '');

        // Detailed validation with specific error messages
        $errors = [];
        if (empty($merchant)) {
            $errors[] = 'Merchant address is missing or empty';
        }
        if (empty($customer)) {
            $errors[] = 'Customer address is missing or empty';
        }
        if ($usd <= 0) {
            $errors[] = 'USD price is missing, zero, or invalid (value: ' . ($_POST['usdprice'] ?? 'NOT SET') . ')';
        }
        if (empty($prod)) {
            $errors[] = 'Product name is missing or empty';
        }
        
        if (!empty($errors)) {
            if (WP_DEBUG && WP_DEBUG_LOG) {
                error_log('Validation Errors: ' . implode(', ', $errors));
            }
            wp_send_json_error(['message' => 'Missing or invalid fields: ' . implode(', ', $errors)]);
        }

        // Build real transaction via Anvil API
        $response = \CardanoMintPay\Helpers\AnvilAPI::buildPaymentTransaction(
            $merchant, 
            $customer, 
            $usd, 
            $prod, 
            'pay'
        );
        
        if (is_wp_error($response)) {
            if (WP_DEBUG && WP_DEBUG_LOG) {
                error_log('Anvil API Error: ' . $response->get_error_message());
            }
            wp_send_json_error(['message' => 'Transaction build failed. Please try again.']);
        }
        
        wp_send_json_success(['transaction' => $response]);
    }

    public static function ajaxSubmitTx() {
        check_ajax_referer('cardanocheckoutnonce', 'nonce');
        
        $transaction = sanitize_text_field($_POST['transaction'] ?? '');
        $signatures = json_decode(stripslashes($_POST['signatures'] ?? '[]'), true);
        
        if (empty($transaction)) {
            wp_send_json_error(['message' => 'Transaction data missing']);
        }
        
        // Submit real transaction via Anvil API
        $response = \CardanoMintPay\Helpers\AnvilAPI::submitTransaction($transaction, $signatures, 'pay');
        
        if (is_wp_error($response)) {
            if (WP_DEBUG && WP_DEBUG_LOG) {
                error_log('Anvil API Submit Error: ' . $response->get_error_message());
            }
            wp_send_json_error(['message' => 'Transaction submission failed. Please try again.']);
        }
        
        wp_send_json_success($response);
    }

    public static function ajaxGetPrice() {
        check_ajax_referer('cardanocheckoutnonce', 'nonce');
        wp_send_json_success(['price' => self::getAdaPrice()]);
    }

    private static function getAdaPrice() {
        $cached = get_transient('cardanocheckoutadaprice');
        if ($cached !== false) {
            return floatval($cached);
        }
        $resp = wp_remote_get('https://api.coingecko.com/api/v3/simple/price?ids=cardano&vs_currencies=usd', ['timeout' => 10]);
        $price = 1.0; // fallback
        if (!is_wp_error($resp)) {
            $body = wp_remote_retrieve_body($resp);
            $data = json_decode($body, true);
            if (isset($data['cardano']['usd'])) {
                $price = floatval($data['cardano']['usd']);
                set_transient('cardanocheckoutadaprice', $price, 300);
            }
        }
        return $price;
    }
}
