<?php
namespace CardanoMintPay\Controllers;

use CardanoMintPay\Helpers\AnvilAPI;

class AJAXController
{
    public static function register()
    {
        // Cardano Pay endpoints
        add_action('wp_ajax_cardanobuildtransaction', [self::class, 'buildTransaction']);
        add_action('wp_ajax_nopriv_cardanobuildtransaction', [self::class, 'buildTransaction']);
        add_action('wp_ajax_cardanosubmittransaction', [self::class, 'submitTransaction']);
        add_action('wp_ajax_nopriv_cardanosubmittransaction', [self::class, 'submitTransaction']);
        add_action('wp_ajax_cardanogetadaprice', [self::class, 'getAdaPrice']);
        add_action('wp_ajax_nopriv_cardanogetadaprice', [self::class, 'getAdaPrice']);

        // Cardano NFT endpoints (add/adjust as in the nft checkout plugin)
        // add_action('wp_ajax_cardanonftmint', [self::class, 'mintNft']);
        // add_action('wp_ajax_nopriv_cardanonftmint', [self::class, 'mintNft']);
        // ... Add further AJAX endpoints as needed from your NFT logic ...

        // Order submission endpoint
        add_action('wp_ajax_cardano_submit_order', [self::class, 'submitOrder']);
        add_action('wp_ajax_nopriv_cardano_submit_order', [self::class, 'submitOrder']);
        
        // Order details endpoint
        add_action('wp_ajax_cardano_get_order_details', [self::class, 'getOrderDetails']);
        add_action('wp_ajax_nopriv_cardano_get_order_details', [self::class, 'getOrderDetails']);
        
        // Test endpoint for Anvil API
        add_action('wp_ajax_cardano_test_anvil_api', [self::class, 'testAnvilAPI']);
        add_action('wp_ajax_nopriv_cardano_test_anvil_api', [self::class, 'testAnvilAPI']);
    }

    public static function buildTransaction()
    {
        check_ajax_referer('cardanocheckoutnonce', 'nonce');
        
        // Rate limiting: max 30 requests per minute per IP (allows for retries and multiple users)
        if (!self::checkRateLimit('build_transaction', 30, 60)) {
            wp_send_json_error(['message' => 'Too many requests. Please wait before trying again.']);
        }
        $merchant = sanitize_text_field($_POST['merchantaddress'] ?? '');
        $customer = sanitize_text_field($_POST['customeraddress'] ?? '');
        $usd = floatval($_POST['usdprice'] ?? 0);
        $prod = sanitize_text_field($_POST['productname'] ?? '');

        if (!$merchant || !$customer || $usd <= 0) {
            wp_send_json_error(['message' => 'Missing or invalid fields.']);
        }

        // Build transaction via Anvil API
        $response = AnvilAPI::buildPaymentTransaction($merchant, $customer, $usd, $prod, 'pay');
        
        if (is_wp_error($response)) {
            if (WP_DEBUG && WP_DEBUG_LOG) {
                error_log('Anvil API Build Error: ' . $response->get_error_message());
            }
            wp_send_json_error(['message' => 'Transaction build failed. Please try again.']);
        }
        
        wp_send_json_success($response);
    }

    public static function submitTransaction()
    {
        check_ajax_referer('cardanocheckoutnonce', 'nonce');
        
        // Rate limiting: max 20 requests per minute per IP (allows for retries and multiple users)
        if (!self::checkRateLimit('submit_transaction', 20, 60)) {
            wp_send_json_error(['message' => 'Too many requests. Please wait before trying again.']);
        }
        
        $transaction = $_POST['transaction'] ?? ''; // Don't sanitize - we need the raw CBOR data
        $signatures = json_decode(stripslashes($_POST['signatures'] ?? '[]'), true);
        
        if (WP_DEBUG && WP_DEBUG_LOG) {
            error_log('=== Transaction Submission Debug ===');
            error_log('Transaction length: ' . strlen($transaction));
            error_log('Transaction preview: ' . substr($transaction, 0, 100) . '...');
            error_log('Signatures: ' . print_r($signatures, true));
        }
        
        if (!$transaction) {
            if (WP_DEBUG && WP_DEBUG_LOG) {
                error_log('Transaction data missing');
            }
            wp_send_json_error(['message' => 'Transaction data missing']);
        }
        
        // Submit transaction via Anvil API
        $response = AnvilAPI::submitTransaction($transaction, $signatures, 'pay');
        
        if (is_wp_error($response)) {
            if (WP_DEBUG && WP_DEBUG_LOG) {
                error_log('Anvil API Submit Error: ' . $response->get_error_message());
            }
            wp_send_json_error(['message' => 'Transaction submission failed. Please try again.']);
        }
        
        if (WP_DEBUG && WP_DEBUG_LOG) {
            error_log('Transaction submitted successfully: ' . print_r($response, true));
        }
        wp_send_json_success($response);
    }

    public static function getAdaPrice()
    {
        check_ajax_referer('cardanocheckoutnonce', 'nonce');
        wp_send_json_success(['price' => AnvilAPI::getAdaPrice()]);
    }

    public static function submitOrder()
    {
        check_ajax_referer('cardanocheckoutnonce', 'nonce');
        
        // Rate limiting: max 15 requests per minute per IP (allows for retries and multiple users)
        if (!self::checkRateLimit('submit_order', 15, 60)) {
            wp_send_json_error(['message' => 'Too many requests. Please wait before trying again.']);
        }
        
        // Debug logging (only in development)
        if (WP_DEBUG && WP_DEBUG_LOG) {
            error_log('=== Order Submission Debug ===');
            error_log('POST data: ' . print_r($_POST, true));
        }
        
        // Validate required fields
        $required_fields = [
            'first_name', 'last_name', 'email', 'phone',
            'shipping_address_1', 'city', 'state_province', 'postal_code', 'country',
            'product_id', 'product_name', 'quantity', 'tx_hash'
        ];
        
        foreach ($required_fields as $field) {
            if (empty($_POST[$field])) {
                if (WP_DEBUG && WP_DEBUG_LOG) {
                    error_log("Missing required field: $field");
                }
                wp_send_json_error(['message' => "Missing required field: $field"]);
            }
        }
        
        // Validate transaction hash (must be real, not simulated)
        $tx_hash = sanitize_text_field($_POST['tx_hash']);
        
        if (WP_DEBUG && WP_DEBUG_LOG) {
            error_log('Transaction hash received: ' . $tx_hash);
        }
        
        if (strpos($tx_hash, 'SIMULATED_') === 0) {
            if (WP_DEBUG && WP_DEBUG_LOG) {
                error_log('Rejected simulated transaction hash');
            }
            wp_send_json_error(['message' => 'Payment not completed. Please complete the payment transaction first.']);
        }
        
        // Prepare order data
        $orderData = [
            'order_id' => sanitize_text_field($_POST['order_id'] ?? ''),
            'order_date' => sanitize_text_field($_POST['order_date'] ?? self::getNewYorkTime()),
            'first_name' => sanitize_text_field($_POST['first_name']),
            'last_name' => sanitize_text_field($_POST['last_name']),
            'email' => sanitize_email($_POST['email']),
            'phone' => sanitize_text_field($_POST['phone']),
            'shipping_address_1' => sanitize_text_field($_POST['shipping_address_1']),
            'shipping_address_2' => sanitize_text_field($_POST['shipping_address_2'] ?? ''),
            'city' => sanitize_text_field($_POST['city']),
            'state_province' => sanitize_text_field($_POST['state_province']),
            'postal_code' => sanitize_text_field($_POST['postal_code']),
            'country' => sanitize_text_field($_POST['country']),
            'special_instructions' => sanitize_textarea_field($_POST['special_instructions'] ?? ''),
            'product_id' => sanitize_text_field($_POST['product_id']),
            'product_name' => sanitize_text_field($_POST['product_name']),
            'quantity' => intval($_POST['quantity']),
            'order_total_usd' => floatval($_POST['order_total_usd'] ?? 0),
            'order_total_ada' => floatval($_POST['order_total_ada'] ?? 0),
            'tx_hash' => $tx_hash,
            'status' => 'completed' // Payment verified, order is complete
        ];
        
        // Insert order into database
        if (WP_DEBUG && WP_DEBUG_LOG) {
            error_log('Inserting order into database...');
        }
        $orderId = \CardanoMintPay\Models\OrderModel::insert($orderData);
        
        if (WP_DEBUG && WP_DEBUG_LOG) {
            error_log('Order insert result: ' . ($orderId ? 'SUCCESS' : 'FAILED'));
        }
        
        if ($orderId) {
            // Send email receipt ONLY after payment verification
            if (WP_DEBUG && WP_DEBUG_LOG) {
                error_log('Sending email receipt...');
            }
            self::sendOrderEmailReceipt($orderData);
            
            if (WP_DEBUG && WP_DEBUG_LOG) {
                error_log('Email receipt sent');
                error_log('Order submission completed successfully');
            }
            wp_send_json_success([
                'order_id' => $orderData['order_id'],
                'tx_hash' => $tx_hash,
                'message' => 'Payment successful! Order completed.',
                'redirect_url' => home_url('/product-order-submitted/?order_id=' . urlencode($orderData['order_id']))
            ]);
        } else {
            if (WP_DEBUG && WP_DEBUG_LOG) {
                error_log('Failed to save order to database');
            }
            wp_send_json_error(['message' => 'Failed to save order']);
        }
    }

    private static function sendOrderEmailReceipt($orderData) {
        // Get order data for email template
        $order = [
            'orderId' => $orderData['order_id'],
            'orderDate' => $orderData['order_date'],
            'firstName' => $orderData['first_name'],
            'lastName' => $orderData['last_name'],
            'email' => $orderData['email'],
            'phone' => $orderData['phone'],
            'shippingAddress1' => $orderData['shipping_address_1'],
            'shippingAddress2' => $orderData['shipping_address_2'],
            'city' => $orderData['city'],
            'stateProvince' => $orderData['state_province'],
            'postalCode' => $orderData['postal_code'],
            'country' => $orderData['country'],
            'specialInstructions' => $orderData['special_instructions'],
            'productId' => $orderData['product_id'],
            'productName' => $orderData['product_name'],
            'quantity' => $orderData['quantity'],
            'orderTotalUsd' => $orderData['order_total_usd'],
            'orderTotalAda' => $orderData['order_total_ada'],
            'status' => $orderData['status'],
            'txHash' => $orderData['tx_hash']
        ];

        // Generate email content
        ob_start();
        include plugin_dir_path(__FILE__) . '/../views/modal-product-order-email-receipt.php';
        $emailContent = ob_get_clean();

        // Send email
        $to = $orderData['email'];
        $subject = 'Your Art Collectors List Order';
        $headers = [
            'Content-Type: text/html; charset=UTF-8',
            'From: ' . get_bloginfo('name') . ' <' . get_option('admin_email') . '>'
        ];

        if (WP_DEBUG && WP_DEBUG_LOG) {
            error_log('=== Email Sending Debug ===');
            error_log('Email to: ' . $to);
            error_log('Email subject: ' . $subject);
            error_log('Email content length: ' . strlen($emailContent));
            error_log('Admin email: ' . get_option('admin_email'));
        }
        
        $emailResult = wp_mail($to, $subject, $emailContent, $headers);
        
        if (WP_DEBUG && WP_DEBUG_LOG) {
            error_log('Email send result: ' . ($emailResult ? 'SUCCESS' : 'FAILED'));
            
            if (!$emailResult) {
                error_log('Email sending failed - check WordPress mail configuration');
            }
        }
    }

    public static function getOrderDetails() {
        check_ajax_referer('cardanocheckoutnonce', 'nonce');
        
        $order_id = sanitize_text_field($_POST['order_id'] ?? '');
        
        if (empty($order_id)) {
            wp_send_json_error(['message' => 'Order ID is required']);
        }
        
        $order = \CardanoMintPay\Models\OrderModel::getByOrderId($order_id);
        
        if (!$order) {
            wp_send_json_error(['message' => 'Order not found']);
        }
        
        // Format order data for display
        $orderData = [
            'orderId' => $order['orderId'],
            'orderDate' => date('F j, Y \a\t g:i A', strtotime($order['orderDate'])),
            'firstName' => $order['firstName'],
            'lastName' => $order['lastName'],
            'email' => $order['email'],
            'phone' => $order['phone'],
            'shippingAddress1' => $order['shippingAddress1'],
            'shippingAddress2' => $order['shippingAddress2'],
            'city' => $order['city'],
            'stateProvince' => $order['stateProvince'],
            'postalCode' => $order['postalCode'],
            'country' => $order['country'],
            'specialInstructions' => $order['specialInstructions'],
            'productId' => $order['productId'],
            'productName' => $order['productName'],
            'quantity' => $order['quantity'],
            'orderTotalUsd' => number_format($order['orderTotalUsd'], 2),
            'orderTotalAda' => number_format($order['orderTotalAda'], 6),
            'status' => ucfirst($order['status']),
            'txHash' => $order['txHash'] ?? 'N/A'
        ];
        
        wp_send_json_success($orderData);
    }

    public static function testAnvilAPI() {
        check_ajax_referer('cardanocheckoutnonce', 'nonce');

        // Test API configuration - check both mint and pay option names
        $api_key = get_option('cardano_mint_anvil_api_key', get_option('cardano_pay_anvil_api_key', ''));
        $api_url = get_option('cardano_mint_anvil_api_url', get_option('cardano_pay_anvil_api_url', ''));
        $merchant_address = get_option('cardano_mint_merchant_address', get_option('cardano_pay_merchant_address', ''));
        
        $test_results = [
            'api_key_configured' => !empty($api_key),
            'api_url_configured' => !empty($api_url),
            'merchant_address_configured' => !empty($merchant_address),
            'api_url' => $api_url,
            'merchant_address' => $merchant_address
        ];
        
        // Test a simple API call
        if (!empty($api_key) && !empty($api_url)) {
            try {
                $response = wp_remote_get($api_url . '/health', [
                    'headers' => [
                        'X-Api-Key' => $api_key
                    ],
                    'timeout' => 10
                ]);
                
                if (!is_wp_error($response)) {
                    $test_results['api_health_check'] = 'success';
                    $test_results['api_response_code'] = wp_remote_retrieve_response_code($response);
                } else {
                    $test_results['api_health_check'] = 'error';
                    $test_results['api_error'] = $response->get_error_message();
                }
            } catch (Exception $e) {
                $test_results['api_health_check'] = 'error';
                $test_results['api_error'] = $e->getMessage();
            }
        } else {
            $test_results['api_health_check'] = 'skipped';
        }
        
        wp_send_json_success($test_results);
    }
    
    /**
     * Get current time in New York timezone
     */
    private static function getNewYorkTime() {
        // Force UTC first, then convert to New York
        $utc = new \DateTimeZone('UTC');
        $ny = new \DateTimeZone('America/New_York');
        
        $datetime = new \DateTime('now', $utc);
        $datetime->setTimezone($ny);
        
        return $datetime->format('Y-m-d H:i:s');
    }
    
    /**
     * Check rate limit for API calls
     */
    private static function checkRateLimit($action, $max_requests, $time_window) {
        $ip = $_SERVER['REMOTE_ADDR'] ?? 'unknown';
        $key = "rate_limit_{$action}_{$ip}";
        
        $requests = get_transient($key);
        if ($requests === false) {
            $requests = 0;
        }
        
        if ($requests >= $max_requests) {
            return false;
        }
        
        set_transient($key, $requests + 1, $time_window);
        return true;
    }

    // Example method for future NFT mint AJAX endpoint:
    /*
    public static function mintNft()
    {
        check_ajax_referer('cardanocheckoutnonce', 'nonce');
        $customer = sanitize_text_field($_POST['customeraddress'] ?? '');
        $policyId = sanitize_text_field($_POST['policyid'] ?? '');
        // Add all minting, metadata handling, DB, etc, as per original NFT logic.
        // wp_send_json_success([...]);
        // wp_send_json_error([...]);
    }
    */
}
