<?php
/*
Plugin Name: Cardano Minting and Payments Engine
Description: Accept ADA payments using CardanoPress and Blockfrost.
Version: 2.0.0
Author: PB_Anvil
Text Domain: cardano-mint-pay
*/
if (!defined('ABSPATH')) {
    exit;
}

// Include required files
require_once plugin_dir_path(__FILE__) . 'includes/helpers/AnvilAPI.php';
require_once plugin_dir_path(__FILE__) . 'includes/models/ProductModel.php';
require_once plugin_dir_path(__FILE__) . 'includes/models/TransactionModel.php';
require_once plugin_dir_path(__FILE__) . 'includes/models/OrderModel.php';
require_once plugin_dir_path(__FILE__) . 'includes/controllers/CheckoutController.php';
require_once plugin_dir_path(__FILE__) . 'includes/controllers/AJAXController.php';

// Register activation hook for database tables
register_activation_hook(__FILE__, 'cardanopay_activate');

function cardanopay_activate() {
    // Create database tables
    CardanoMintPay\Models\TransactionModel::installTable();
    CardanoMintPay\Models\OrderModel::installTable();
    
    // Create products table
    global $wpdb;
    $table = $wpdb->prefix . 'cardanopayproducts';
    $charset_collate = $wpdb->get_charset_collate();
    $sql = "CREATE TABLE IF NOT EXISTS $table (
        ProductID varchar(50) NOT NULL,
        ProductImage varchar(255),
        ProductName varchar(255) NOT NULL,
        ProductDescription text,
        ProductHeight varchar(50),
        ProductWidth varchar(50),
        ProductPrice decimal(10,2) NOT NULL DEFAULT 0,
        QuantityLimitPerCustomer int(11) NOT NULL DEFAULT 1,
        PRIMARY KEY (ProductID)
    ) $charset_collate;";
    require_once(ABSPATH . 'wp-admin/includes/upgrade.php');
    dbDelta($sql);
}

// Register deactivation hook for cleanup
register_deactivation_hook(__FILE__, 'cardanopay_deactivate');

function cardanopay_deactivate() {
    $delete_data = get_option('cardano-pay-delete_data_on_deactivation', 0);
    
    if ($delete_data) {
        global $wpdb;
        
        // Drop plugin tables
        $wpdb->query("DROP TABLE IF EXISTS {$wpdb->prefix}cardanopayproducts");
        $wpdb->query("DROP TABLE IF EXISTS {$wpdb->prefix}cardanopay_orders");
        $wpdb->query("DROP TABLE IF EXISTS {$wpdb->prefix}cardanopay_transactions");
        
        // Delete plugin options
        $wpdb->query("DELETE FROM {$wpdb->options} WHERE option_name LIKE 'cardano-pay-%'");
        
        // Clear any cached data
        wp_cache_flush();
    }
}

// Initialize controllers
add_action('init', function() {
    CardanoMintPay\Controllers\CheckoutController::register();
    CardanoMintPay\Controllers\AJAXController::register();
});

// Register product order receipt shortcode
add_shortcode('product-order-receipt', 'cardanopay_product_order_receipt_shortcode');

function cardanopay_product_order_receipt_shortcode($atts) {
    $atts = shortcode_atts([
        'order-id' => '',
        'order_id' => ''
    ], $atts);
    
    $orderId = !empty($atts['order-id']) ? $atts['order-id'] : $atts['order_id'];
    
    if (empty($orderId)) {
        return '<p>Order ID is required to display receipt.</p>';
    }
    
    $order = CardanoMintPay\Models\OrderModel::getByOrderId($orderId);
    
    if (!$order) {
        return '<p>Order not found. Please check your order ID.</p>';
    }
    
    ob_start();
    include plugin_dir_path(__FILE__) . 'includes/views/modal-product-order-email-receipt.php';
    return ob_get_clean();
}

// Enqueue scripts and styles
add_action('wp_enqueue_scripts', function() {
    global $post;
    
    $load_assets = false;
    
    // Original Gutenberg detection (preserves existing functionality)
    if (is_a($post, 'WP_Post') && has_shortcode($post->post_content, 'cardano-pay')) {
        $load_assets = true;
    }
    
    // Bricks Builder detection
    if (!$load_assets) {
        // Check if Bricks Builder is active and we're on a frontend page
        if (defined('BRICKS_VERSION')) {
            // On singular pages where Bricks might be used
            if (is_singular()) {
                $load_assets = true;
            }
            // During Bricks builder preview/editing
            if (isset($_GET['bricks']) || (function_exists('bricks_is_builder_main_query') && bricks_is_builder_main_query())) {
                $load_assets = true;
            }
        }
    }
    
    // Additional fallback for other page builders or manual loading
    if (!$load_assets) {
        // Check if we're on a page that might contain the shortcode
        if (is_page() || is_single()) {
            // Load assets on all pages/posts as fallback (can be optimized later)
            $load_assets = true;
        }
    }
    
    // Load assets if any condition is met
    if ($load_assets) {
        // Prevent duplicate loading
        if (!wp_script_is('cardano-checkout-js', 'enqueued')) {
        // No external wallet library needed - using direct CIP-30 implementation
            
            // Our plugin script (uses direct CIP-30 wallet integration)
            wp_enqueue_script(
                'cardano-checkout-js', 
                plugin_dir_url(__FILE__) . 'assets/cardano-checkout.js', 
                ['jquery'], 
                '1.0.0', 
                true
            );
            
            wp_enqueue_style('cardano-checkout-css', plugin_dir_url(__FILE__) . 'assets/cardano-checkout.css', [], '1.0.0');
            
            // Localize script for AJAX
            wp_localize_script('cardano-checkout-js', 'cardanoCheckout', [
                'ajaxurl' => admin_url('admin-ajax.php'),
                'nonce' => wp_create_nonce('cardanocheckoutnonce'),
                'network' => cardanopay_get_network_name(),
                'apiConfigured' => !empty(get_option('cardano_pay_anvil_api_key', '')),
                'merchantConfigured' => !empty(get_option('cardano_pay_merchant_address', '')),
                'home_url' => home_url(),
                'debug' => WP_DEBUG && WP_DEBUG_LOG
            ]);
        }
    }
});

// Get network name for explorer links
function cardanopay_get_network_name() {
    $api_url = get_option('cardano_pay_anvil_api_url', 'https://preprod.api.ada-anvil.app/v2/services');
    
    if (strpos($api_url, 'preprod') !== false) {
        return 'preprod';
    } elseif (strpos($api_url, 'preview') !== false) {
        return 'preview';
    } else {
        return 'mainnet';
    }
}

add_action('admin_menu', 'cardanopay_admin_menu');

// Handle CSV export before any admin output
add_action('admin_init', 'cardanopay_handle_csv_export');

function cardanopay_handle_csv_export() {
    if (isset($_GET['page']) && $_GET['page'] === 'cardano-pay-orders' && isset($_GET['export_csv'])) {
        // CSRF protection for CSV export
        if (!current_user_can('manage_options')) {
            wp_die('Insufficient permissions.');
        }
        
        // Additional nonce verification for CSV export
        if (!wp_verify_nonce($_GET['_wpnonce'] ?? '', 'export_csv_orders')) {
            wp_die('Security check failed. Please try again.');
        }
        
        cardanopay_export_orders_csv();
        exit;
    }
}

function cardanopay_admin_menu() {
    add_menu_page(
        'Cardano Pay',
        'Cardano Pay',
        'manage_options',
        'cardano-pay-plugin-setup',
        'cardanopay_setup_page',
        'dashicons-money-alt',
        55
    );

    add_submenu_page(
        'cardano-pay-plugin-setup',
        'Product Manager',
        'Product Manager',
        'manage_options',
        'cardano-pay-product-manager',
        'cardanopay_product_manager_page'
    );
    add_submenu_page(
        'cardano-pay-plugin-setup',
        'Product Inventory List',
        'Product Inventory List',
        'manage_options',
        'cardano-pay-inventory-list',
        'cardanopay_product_inventory_list_page'
    );
    add_submenu_page(
        'cardano-pay-plugin-setup',
        'Product Orders',
        'Product Orders',
        'manage_options',
        'cardano-pay-orders',
        'cardanopay_product_orders_page'
    );
    
    add_submenu_page(
        'cardano-pay-plugin-setup',
        'How to Use',
        'How to Use',
        'manage_options',
        'cardano-pay-how-to-use',
        'cardanopay_how_to_use_page'
    );
}

function cardanopay_setup_page() {
    $prefix = 'cardano-pay';
    $settingsmessage = '';

    if (isset($_POST['pluginsetupsave'])) {
        check_admin_referer('pluginsetupoptions');
        
        // Additional CSRF protection
        if (!wp_verify_nonce($_POST['_wpnonce'], 'pluginsetupoptions')) {
            wp_die('Security check failed. Please try again.');
        }
        
        // Determine API URL based on network selection
        $network = sanitize_text_field($_POST['networkenvironment']);
        $api_url = ($network === 'mainnet') ? 'https://prod.api.ada-anvil.app/v2/services' : 'https://preprod.api.ada-anvil.app/v2/services';
        
        // Use appropriate API key based on network
        $api_key = ($network === 'mainnet') ? sanitize_text_field($_POST['anvilapikeymainnet']) : sanitize_text_field($_POST['anvilapikeytestnet']);
        
        // Save standardized options for AnvilAPI helper
        update_option('cardano_pay_anvil_api_key', $api_key);
        update_option('cardano_pay_anvil_api_url', $api_url);
        update_option('cardano_pay_merchant_address', sanitize_text_field($_POST['merchantwalletaddress']));
        
        // Keep legacy options for backward compatibility
        update_option($prefix . '-anvilapikeymainnet', sanitize_text_field($_POST['anvilapikeymainnet']));
        update_option($prefix . '-anvilapikeytestnet', sanitize_text_field($_POST['anvilapikeytestnet']));
        update_option($prefix . '-merchantwalletaddress', sanitize_text_field($_POST['merchantwalletaddress']));
        update_option($prefix . '-networkenvironment', $network);
        update_option($prefix . '-delete_data_on_deactivation', isset($_POST['delete_data_on_deactivation']) ? 1 : 0);
        
        $settingsmessage = 'Settings saved!';
    }

    $anvilapikeymainnet = get_option($prefix . '-anvilapikeymainnet', '');
    $anvilapikeytestnet = get_option($prefix . '-anvilapikeytestnet', '');
    $merchantwalletaddress = get_option($prefix . '-merchantwalletaddress', '');
    $networkenvironment = get_option($prefix . '-networkenvironment', 'mainnet');
    $delete_data_on_deactivation = get_option($prefix . '-delete_data_on_deactivation', 0);

    ?>
    <div class="wrap">
        <h1>Cardano Pay Plugin Setup</h1>
        <?php if ($settingsmessage): ?>
            <div class="notice notice-success">
                <p><?php echo esc_html($settingsmessage); ?></p>
            </div>
        <?php endif; ?>
        <form method="post">
            <?php wp_nonce_field('pluginsetupoptions'); ?>
            <table class="form-table">
                <tr>
                    <th>ADA Anvil API Key Mainnet</th>
                    <td>
                        <input type="text" name="anvilapikeymainnet" value="<?php echo esc_attr($anvilapikeymainnet); ?>" size="60" />
                    </td>
                </tr>
                <tr>
                    <th>ADA Anvil API Key Testnet</th>
                    <td>
                        <input type="text" name="anvilapikeytestnet" value="<?php echo esc_attr($anvilapikeytestnet); ?>" size="60" />
                    </td>
                </tr>
                <tr>
                    <th>Merchant Wallet Address</th>
                    <td>
                        <input type="text" name="merchantwalletaddress" value="<?php echo esc_attr($merchantwalletaddress); ?>" size="60" />
                    </td>
                </tr>
                <tr>
                    <th>Network Environment</th>
                    <td>
                        <select name="networkenvironment">
                            <option value="mainnet" <?php selected($networkenvironment, 'mainnet', true); ?>>Mainnet</option>
                            <option value="preprod" <?php selected($networkenvironment, 'preprod', true); ?>>Preprod Testnet</option>
                        </select>
                    </td>
                </tr>
                <tr>
                    <th>Data Cleanup</th>
                    <td>
                        <input type="checkbox" name="delete_data_on_deactivation" id="delete_data_on_deactivation" value="1" <?php checked($delete_data_on_deactivation, 1); ?> />
                        <label for="delete_data_on_deactivation">Delete all plugin data when deactivated</label>
                        <p class="description">This will remove all products, orders, transactions, and settings when the plugin is deactivated.</p>
                    </td>
                </tr>
            </table>
            <p class="submit">
                <input type="submit" name="pluginsetupsave" class="button-primary" value="Save Settings" />
                <button type="button" id="test-anvil-api" class="button" style="margin-left: 10px;">Test Anvil API Connection</button>
            </p>
            
            <div id="anvil-test-results" style="margin-top: 20px; padding: 15px; border-radius: 5px; display: none;">
                <h3>Anvil API Test Results</h3>
                <div id="test-results-content"></div>
            </div>
            
            <script>
            document.getElementById('test-anvil-api').addEventListener('click', function() {
                const button = this;
                const resultsDiv = document.getElementById('anvil-test-results');
                const contentDiv = document.getElementById('test-results-content');
                
                button.disabled = true;
                button.textContent = 'Testing...';
                resultsDiv.style.display = 'block';
                contentDiv.innerHTML = '<p>Testing Anvil API connection...</p>';
                
                const formData = new FormData();
                formData.append('action', 'cardano_test_anvil_api');
                formData.append('nonce', '<?php echo wp_create_nonce('cardanocheckoutnonce'); ?>');
                
                fetch('<?php echo admin_url('admin-ajax.php'); ?>', {
                    method: 'POST',
                    body: formData
                })
                .then(response => response.json())
                .then(data => {
                    if (data.success) {
                        const results = data.data;
                        let html = '<ul>';
                        html += '<li><strong>API Key:</strong> ' + (results.api_key_configured ? '✅ Configured' : '❌ Missing') + '</li>';
                        html += '<li><strong>API URL:</strong> ' + (results.api_url_configured ? '✅ Configured' : '❌ Missing') + '</li>';
                        html += '<li><strong>Merchant Address:</strong> ' + (results.merchant_address_configured ? '✅ Configured' : '❌ Missing') + '</li>';
                        html += '<li><strong>API URL:</strong> ' + results.api_url + '</li>';
                        html += '<li><strong>Merchant Address:</strong> ' + results.merchant_address + '</li>';
                        
                        if (results.api_health_check === 'success') {
                            html += '<li><strong>API Health Check:</strong> ✅ Success (HTTP ' + results.api_response_code + ')</li>';
                        } else if (results.api_health_check === 'error') {
                            html += '<li><strong>API Health Check:</strong> ❌ Error - ' + results.api_error + '</li>';
                        } else {
                            html += '<li><strong>API Health Check:</strong> ⏭️ Skipped (missing configuration)</li>';
                        }
                        
                        html += '</ul>';
                        contentDiv.innerHTML = html;
                    } else {
                        contentDiv.innerHTML = '<p style="color: red;">Test failed: ' + (data.data?.message || 'Unknown error') + '</p>';
                    }
                })
                .catch(error => {
                    contentDiv.innerHTML = '<p style="color: red;">Test failed: ' + error.message + '</p>';
                })
                .finally(() => {
                    button.disabled = false;
                    button.textContent = 'Test Anvil API Connection';
                });
            });
            </script>
        </form>
    </div>
    <?php
}

function cardanopay_product_manager_page() {
    $message = '';
    
    // Handle form submissions
    if (isset($_POST['saveproduct']) && wp_verify_nonce($_POST['_wpnonce'], 'saveproductcardanopay')) {
        // Additional CSRF protection
        if (!current_user_can('manage_options')) {
            wp_die('Insufficient permissions.');
        }
        $productId = sanitize_text_field($_POST['ProductID'] ?? '');
        if (empty($productId)) {
            $productId = uniqid('prod_');
        }
        
        $productData = [
            'ProductID' => $productId,
            'ProductImage' => sanitize_text_field($_POST['ProductImage'] ?? ''),
            'ProductName' => sanitize_text_field($_POST['ProductName'] ?? ''),
            'ProductDescription' => sanitize_textarea_field($_POST['ProductDescription'] ?? ''),
            'ProductHeight' => sanitize_text_field($_POST['ProductHeight'] ?? ''),
            'ProductWidth' => sanitize_text_field($_POST['ProductWidth'] ?? ''),
            'ProductPrice' => floatval($_POST['ProductPrice'] ?? 0),
            'QuantityLimitPerCustomer' => intval($_POST['QuantityLimitPerCustomer'] ?? 1)
        ];
        
        $result = CardanoMintPay\Models\ProductModel::createProduct($productData);
        if ($result) {
            $message = 'Product saved successfully!';
        } else {
            $message = 'Error saving product.';
        }
    }
    
    include plugin_dir_path(__FILE__) . 'includes/views/product-manager.php';
}

function cardanopay_product_inventory_list_page() {
    $products = CardanoMintPay\Models\ProductModel::listProducts(10);
    $slots_used = count($products);
    $admin_url = admin_url();
    
    // Handle bulk delete
    if (isset($_POST['bulkdelete']) && wp_verify_nonce($_POST['_wpnonce'], 'bulkdeleteproducts')) {
        // Additional CSRF protection
        if (!current_user_can('manage_options')) {
            wp_die('Insufficient permissions.');
        }
        if (!empty($_POST['selectedproducts'])) {
            foreach ($_POST['selectedproducts'] as $productId) {
                CardanoMintPay\Models\ProductModel::deleteProduct($productId);
            }
            wp_redirect(admin_url('admin.php?page=cardano-pay-inventory-list&deleted=1'));
            exit;
        }
    }
    
    // Product updates removed - table is now read-only for security
    
    // Handle individual product delete
    if (isset($_GET['deleteitem'])) {
        // CSRF protection for GET requests
        if (!current_user_can('manage_options')) {
            wp_die('Insufficient permissions.');
        }
        $deleted = CardanoMintPay\Models\ProductModel::deleteProduct($_GET['deleteitem']);
        if ($deleted) {
            wp_redirect(admin_url('admin.php?page=cardano-pay-inventory-list&deleted=1'));
            exit;
        }
    }
    
    // Reload products after any changes
    $products = CardanoMintPay\Models\ProductModel::listProducts(10);
    $slots_used = count($products);
    
    include plugin_dir_path(__FILE__) . 'includes/views/product-inventory.php';
}

function cardanopay_product_orders_page() {
    
    $orderby = sanitize_text_field($_GET['orderby'] ?? 'createdAt');
    $order = sanitize_text_field($_GET['order'] ?? 'DESC');
    $page = max(1, intval($_GET['paged'] ?? 1));
    $per_page = 50;
    $offset = ($page - 1) * $per_page;
    
    $total_orders = CardanoMintPay\Models\OrderModel::countOrders();
    $total_pages = ceil($total_orders / $per_page);
    $orders = CardanoMintPay\Models\OrderModel::listOrders($per_page, $offset, $orderby, $order);
    
    // Handle order status updates
    if (isset($_POST['update_status']) && wp_verify_nonce($_POST['_wpnonce'], 'update_order_status')) {
        // Additional CSRF protection
        if (!current_user_can('manage_options')) {
            wp_die('Insufficient permissions.');
        }
        $order_id = sanitize_text_field($_POST['order_id']);
        $status = sanitize_text_field($_POST['status']);
        
        if (WP_DEBUG && WP_DEBUG_LOG) {
            error_log('=== Order Status Update Debug ===');
            error_log('Order ID: ' . $order_id);
            error_log('New Status: ' . $status);
        }
        
        $result = CardanoMintPay\Models\OrderModel::updateStatus($order_id, $status);
        
        if (WP_DEBUG && WP_DEBUG_LOG) {
            error_log('Update result: ' . ($result ? 'SUCCESS' : 'FAILED'));
        }
        
        wp_redirect(admin_url('admin.php?page=cardano-pay-orders&updated=1'));
        exit;
    }
    
    // Handle individual order deletion
    if (isset($_GET['delete_order'])) {
        // CSRF protection for GET requests
        if (!current_user_can('manage_options')) {
            wp_die('Insufficient permissions.');
        }
        CardanoMintPay\Models\OrderModel::deleteOrder($_GET['delete_order']);
        wp_redirect(admin_url('admin.php?page=cardano-pay-orders&deleted=1'));
        exit;
    }
    
    // Handle bulk delete
    if (isset($_POST['bulk_delete_orders']) && wp_verify_nonce($_POST['_wpnonce'], 'bulk_delete_orders')) {
        // Additional CSRF protection
        if (!current_user_can('manage_options')) {
            wp_die('Insufficient permissions.');
        }
        if (!empty($_POST['selected_orders'])) {
            foreach ($_POST['selected_orders'] as $order_id) {
                CardanoMintPay\Models\OrderModel::deleteOrder($order_id);
            }
        }
        wp_redirect(admin_url('admin.php?page=cardano-pay-orders&bulk_deleted=1'));
        exit;
    }
    
    // Handle delete all orders
    if (isset($_POST['delete_all_orders']) && wp_verify_nonce($_POST['_wpnonce'], 'delete_all_orders')) {
        // Additional CSRF protection
        if (!current_user_can('manage_options')) {
            wp_die('Insufficient permissions.');
        }
        CardanoMintPay\Models\OrderModel::deleteAllOrders();
        wp_redirect(admin_url('admin.php?page=cardano-pay-orders&all_deleted=1'));
        exit;
    }
    
    include plugin_dir_path(__FILE__) . 'includes/views/product-orders.php';
}

function cardanopay_export_orders_csv() {
    // Get orders data
    $orders = CardanoMintPay\Models\OrderModel::exportOrders();
    
    $filename = 'product-orders-' . date('Y-m-d-H-i-s') . '.csv';
    
    // Set headers for CSV download
    header('Content-Type: text/csv; charset=UTF-8');
    header('Content-Disposition: attachment; filename="' . $filename . '"');
    header('Pragma: no-cache');
    header('Expires: 0');
    header('Cache-Control: must-revalidate, post-check=0, pre-check=0');
    
    // Add BOM for UTF-8 compatibility with Excel
    echo "\xEF\xBB\xBF";
    
    $output = fopen('php://output', 'w');
    
    // CSV Headers
    fputcsv($output, [
        'Order ID', 'Order Date', 'First Name', 'Last Name', 'Email', 'Phone',
        'Shipping Address 1', 'Shipping Address 2', 'City', 'State/Province',
        'Postal Code', 'Country', 'Special Instructions', 'Product ID', 'Product Name',
        'Quantity', 'Order Total USD', 'Order Total ADA', 'Transaction Hash', 'Status'
    ]);
    
    // CSV Data
    foreach ($orders as $order) {
        fputcsv($output, [
            $order['orderId'],
            $order['orderDate'],
            $order['firstName'],
            $order['lastName'],
            $order['email'],
            $order['phone'],
            $order['shippingAddress1'],
            $order['shippingAddress2'],
            $order['city'],
            $order['stateProvince'],
            $order['postalCode'],
            $order['country'],
            $order['specialInstructions'],
            $order['productId'],
            $order['productName'],
            $order['quantity'],
            $order['orderTotalUsd'],
            $order['orderTotalAda'],
            $order['txHash'] ?? 'N/A',
            $order['status']
        ]);
    }
    
    fclose($output);
    exit;
}

function cardanopay_how_to_use_page() {
    ?>
    <div class="wrap">
        <h1>Cardano Pay - How to Use</h1>
        
        <div style="background: #f9f9f9; padding: 20px; border-radius: 5px; margin-bottom: 20px;">
            <h2>📋 Available Shortcodes</h2>
            <p>The Cardano Pay plugin provides the following shortcode for accepting ADA payments:</p>
        </div>

        <div style="background: white; padding: 20px; border: 1px solid #ddd; border-radius: 5px; margin-bottom: 20px;">
            <h3>🛒 [cardano-pay] - Product Payment Shortcode</h3>
            <p><strong>Description:</strong> Creates a "BUY NOW" button that opens a payment modal for purchasing digital products with ADA.</p>
            <p><strong>IMPORTANT:</strong> This shortcode only works properly inside of Gutenberg Post Types (for example, Blogs). To ensure proper functionality, refrain from placing it within Pages.</p>
            
            <h4>Available Attributes:</h4>
            <table class="wp-list-table widefat fixed striped">
                <thead>
                    <tr>
                        <th style="width: 20%;">Attribute</th>
                        <th style="width: 15%;">Required</th>
                        <th style="width: 15%;">Default</th>
                        <th style="width: 50%;">Description</th>
                    </tr>
                </thead>
                <tbody>
                    <tr>
                        <td><code>product-id</code></td>
                        <td><span style="color: #d63638;">Yes</span></td>
                        <td>-</td>
                        <td>The unique ID of the product from your Product Manager</td>
                    </tr>
                    <tr>
                        <td><code>productname</code></td>
                        <td><span style="color: #00a32a;">No</span></td>
                        <td>"Physical Product"</td>
                        <td>Display name for the product (overrides database value)</td>
                    </tr>
                    <tr>
                        <td><code>price</code></td>
                        <td><span style="color: #d63638;">Yes*</span></td>
                        <td>0</td>
                        <td>Price in USD (overrides database value). <strong>Required if not set in database.</strong></td>
                    </tr>
                    <tr>
                        <td><code>class</code></td>
                        <td><span style="color: #00a32a;">No</span></td>
                        <td>-</td>
                        <td>CSS class for styling</td>
                    </tr>
                </tbody>
            </table>

            <h4>📝 Usage Examples:</h4>
            
            <h5>For Bricks Builder Pages (Default):</h5>
            <div style="background: #f0f0f0; padding: 15px; border-radius: 3px; font-family: monospace; margin: 10px 0;">
                [cardano-pay product-id="prod_123abc" price="25"]
            </div>
            <p><em><strong>Bricks Builder:</strong> Omit the class attribute for default button styling with proper padding.</em></p>


            <h5>Basic Usage (Database Product):</h5>
            <div style="background: #f0f0f0; padding: 15px; border-radius: 3px; font-family: monospace; margin: 10px 0;">
                [cardano-pay product-id="prod_123abc"]
            </div>
            <p><em>Uses product details (including price) from your Product Manager database.</em></p>

            <h5>With Custom Product Name (Bricks):</h5>
            <div style="background: #f0f0f0; padding: 15px; border-radius: 3px; font-family: monospace; margin: 10px 0;">
                [cardano-pay product-id="prod_123abc" productname="Premium Digital Art" price="75"]
            </div>

            <h5>With Custom Product Name (Gutenberg):</h5>
            <div style="background: #f0f0f0; padding: 15px; border-radius: 3px; font-family: monospace; margin: 10px 0;">
                [cardano-pay product-id="prod_123abc" productname="Premium Digital Art" price="75"]
            </div>
        </div>

        <div style="background: #e7f3ff; padding: 20px; border-radius: 5px; border-left: 4px solid #0073aa;">
            <h3>💡 Pro Tips</h3>
            <ul>
                <li><strong>Product ID:</strong> Create products in the "Product Manager" page to get product IDs</li>
                <li><strong>Price Requirement:</strong> Either set a price in the Product Manager OR use the price attribute in the shortcode</li>
                <li><strong>Merchant Address:</strong> Configure your Cardano wallet address in the Plugin Setup page - this will be used for all payments</li>
                <li><strong>Pricing:</strong> Prices are automatically converted from USD to ADA using live exchange rates</li>
                <li><strong>Dual Payments:</strong> Customers pay the full amount to you + receive 1 ADA back as a receipt</li>
                <li><strong>Wallet Support:</strong> Compatible with Eternl, Lace, and other CIP-30 wallets</li>
                <li><strong>Network:</strong> Configure Mainnet/Preprod in Plugin Setup</li>
            </ul>
        </div>


        <div style="background: #fff3cd; padding: 20px; border-radius: 5px; border-left: 4px solid #ffc107; margin-top: 20px;">
            <h3>⚠️ Important Setup Requirements</h3>
            <ol>
                <li><strong>Configure API Keys:</strong> Add your Anvil API keys in "Plugin Setup"</li>
                <li><strong>Set Merchant Address:</strong> Configure your Cardano address to receive payments</li>
                <li><strong>Create Products:</strong> Use "Product Manager" to create products and get product IDs</li>
                <li><strong>Test First:</strong> Use Preprod network for testing before going live</li>
            </ol>
        </div>
    </div>
    <?php
}
?>