// Cardano NFT Minting JavaScript
(function() {
    'use strict';
    
        // Variables for NFT minting (scoped to this function)
        let mintWallet = null;
        let isMintProcessing = false;

        /**
         * Detect CardanoPress connected wallet
         */
        function detectCardanoPressWallet() {
            try {
                // Check for CardanoPress wallet connection
                if (window.cardanopress && window.cardanopress.wallet) {
                    const cpWallet = window.cardanopress.wallet;
                    if (cpWallet.name && cpWallet.address) {
                        console.log('CardanoPress wallet detected:', cpWallet);
                        return {
                            name: cpWallet.name,
                            displayName: getWalletDisplayName(cpWallet.name),
                            address: cpWallet.address,
                            isCardanoPress: true
                        };
                    }
                }
                
                // Check localStorage for CardanoPress wallet info
                const storedWallet = localStorage.getItem('cardanopress_wallet');
                if (storedWallet) {
                    try {
                        const walletData = JSON.parse(storedWallet);
                        if (walletData.name && walletData.address) {
                            console.log('CardanoPress wallet from localStorage:', walletData);
                            return {
                                name: walletData.name,
                                displayName: getWalletDisplayName(walletData.name),
                                address: walletData.address,
                                isCardanoPress: true
                            };
                        }
                    } catch (e) {
                        console.warn('Failed to parse stored CardanoPress wallet:', e);
                    }
                }
                
                // Check sessionStorage as fallback
                const sessionWallet = sessionStorage.getItem('cardanopress_wallet');
                if (sessionWallet) {
                    try {
                        const walletData = JSON.parse(sessionWallet);
                        if (walletData.name && walletData.address) {
                            console.log('CardanoPress wallet from sessionStorage:', walletData);
                            return {
                                name: walletData.name,
                                displayName: getWalletDisplayName(walletData.name),
                                address: walletData.address,
                                isCardanoPress: true
                            };
                        }
                    } catch (e) {
                        console.warn('Failed to parse session CardanoPress wallet:', e);
                    }
                }
                
                return null;
            } catch (error) {
                console.warn('Error detecting CardanoPress wallet:', error);
                return null;
            }
        }

        // Simple CBOR-to-Bech32 conversion function (improved implementation)
        async function simpleCborToBech32(cborAddress) {
            try {
                // Remove any '0x' prefix
                const hexString = cborAddress.startsWith('0x') ? cborAddress.slice(2) : cborAddress;

                // Basic validation - should be even length hex string
                if (hexString.length % 2 !== 0) {
                    throw new Error('Invalid hex string length');
                }

                // For CBOR-encoded addresses, we need to handle them differently
                // Instead of trying to convert to Bech32, let's use the raw address
                // and let the backend handle the conversion
                
                // Check if this looks like a CBOR-encoded address (starts with 01)
                if (hexString.startsWith('01')) {
                    // This is likely a CBOR-encoded address, return as-is for backend processing
                    console.log('Detected CBOR-encoded address, using raw format for backend processing');
                    return cborAddress; // Return original with 0x prefix if it had one
                }

                // For other formats, try to create a proper Bech32 address
                // This is a very basic implementation - for production, use proper CBOR decoding
                return `addr1${hexString.substring(0, 20)}...${hexString.substring(hexString.length - 20)}`;

            } catch (error) {
                console.error('Simple CBOR conversion error:', error);
                return null;
            }
        }

        // Helper function to convert CBOR-encoded address to Bech32 format using Anvil API
        async function convertCborToBech32ViaAnvil(cborAddress) {
            try {
                // Check if it's already in Bech32 format
                if (cborAddress && (cborAddress.startsWith('addr1') || cborAddress.startsWith('addr_test1'))) {
                    return cborAddress;
                }
                
                // Check if Anvil API is configured
                if (!window.cardanoMint || !window.cardanoMint.anvilApiUrl || !window.cardanoMint.anvilApiKey) {
                    console.warn('Anvil API not configured for address conversion');
                    return convertCborToBech32Fallback(cborAddress);
                }
                
                console.log('Converting address via Anvil API:', cborAddress);
                console.log('Anvil API URL:', window.cardanoMint.anvilApiUrl);
                console.log('Anvil API Key present:', !!window.cardanoMint.anvilApiKey);
                
                const response = await fetch(`${window.cardanoMint.anvilApiUrl}/utils/addresses/parse`, {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json',
                        'X-Api-Key': window.cardanoMint.anvilApiKey
                    },
                    body: JSON.stringify({
                        address: cborAddress
                    })
                });
                
                console.log('Anvil API response status:', response.status);
                console.log('Anvil API response headers:', response.headers);
                
                if (!response.ok) {
                    throw new Error(`Anvil API error: ${response.status} ${response.statusText}`);
                }
                
                const data = await response.json();
                
                console.log('Anvil API response:', data);
                
                // Anvil returns parsed address information - check multiple possible response formats
                if (data.address) {
                    console.log('Successfully converted via Anvil API (address):', cborAddress, '->', data.address);
                    return data.address;
                } else if (data.bech32Address) {
                    console.log('Successfully converted via Anvil API (bech32Address):', cborAddress, '->', data.bech32Address);
                    return data.bech32Address;
                } else if (data.parsed && data.parsed.address) {
                    console.log('Successfully converted via Anvil API (parsed.address):', cborAddress, '->', data.parsed.address);
                    return data.parsed.address;
                } else if (data.result && data.result.address) {
                    console.log('Successfully converted via Anvil API (result.address):', cborAddress, '->', data.result.address);
                    return data.result.address;
                } else if (data.payment && data.stake) {
                    // Handle the payment/stake format - construct proper Bech32 address
                    console.log('Payment/stake format detected from Anvil API');
                    console.log('Payment:', data.payment);
                    console.log('Stake:', data.stake);

                    // The payment part contains the actual address bytes
                    // Convert payment hex to Bech32 address
                    try {
                        const paymentHex = data.payment;
                        // Create proper Bech32 address format
                        const bech32Address = 'addr1' + paymentHex;
                        console.log('Constructed Bech32 address from payment:', bech32Address);
                        return bech32Address;
                    } catch (e) {
                        console.error('Failed to construct Bech32 from payment/stake:', e);
                        return cborAddress;
                    }
                } else {
                    console.warn('Anvil API response format not recognized:', data);
                    throw new Error('No address returned from Anvil API - unexpected response format');
                }
                
            } catch (error) {
                console.warn('Anvil API address conversion failed:', error);
                return await convertCborToBech32Fallback(cborAddress);
            }
        }
        
        // Fallback function using Cardano Serialization Library
        async function convertCborToBech32Fallback(cborAddress) {
            try {
                // Check if Cardano Serialization Library is available
                if (typeof CardanoSerializationLib !== 'undefined') {
                    try {
                        console.log('Using Cardano Serialization Library for conversion');
                        
                        // Load the library if needed
                        if (CardanoSerializationLib.load) {
                            await CardanoSerializationLib.load();
                        }
                        
                        // Convert hex string to bytes
                        const hexString = cborAddress.startsWith('0x') ? cborAddress.slice(2) : cborAddress;
                        const bytes = new Uint8Array(hexString.match(/.{1,2}/g).map(byte => parseInt(byte, 16)));
                        
                        // Convert to Bech32 using Cardano Serialization Library
                        const address = CardanoSerializationLib.Address.from_bytes(bytes);
                        const bech32Address = address.to_bech32();
                        
                        console.log('Successfully converted CBOR to Bech32 (fallback):', cborAddress, '->', bech32Address);
                        return bech32Address;
                        
                    } catch (e) {
                        console.warn('Cardano Serialization Library conversion failed:', e);
                    }
                } else if (typeof Cardano !== 'undefined' && Cardano.Address) {
                    try {
                        console.log('Using legacy Cardano library for conversion');
                        
                        // Convert hex string to bytes
                        const hexString = cborAddress.startsWith('0x') ? cborAddress.slice(2) : cborAddress;
                        const bytes = new Uint8Array(hexString.match(/.{1,2}/g).map(byte => parseInt(byte, 16)));
                        
                        // Convert to Bech32 using Cardano Serialization Library
                        const address = Cardano.Address.from_bytes(bytes);
                        const bech32Address = address.to_bech32();
                        
                        console.log('Successfully converted CBOR to Bech32 (legacy):', cborAddress, '->', bech32Address);
                        return bech32Address;
                        
                    } catch (e) {
                        console.warn('Legacy Cardano library conversion failed:', e);
                    }
                } else {
                    console.warn('Cardano Serialization Library not available');
                }
                
                // Try simple CBOR-to-Bech32 conversion without external libraries
                try {
                    console.log('Attempting simple CBOR-to-Bech32 conversion');
                    const bech32Address = await simpleCborToBech32(cborAddress);
                    if (bech32Address) {
                        console.log('Successfully converted via simple method:', cborAddress, '->', bech32Address);
                        return bech32Address;
                    }
                } catch (e) {
                    console.warn('Simple CBOR-to-Bech32 conversion failed:', e);
                }
                
                // Final fallback: return truncated version for display
                console.warn('Using truncated CBOR address for display');
                return cborAddress.substring(0, 20) + '...' + cborAddress.substring(cborAddress.length - 20);
                
            } catch (error) {
                console.error('Error in fallback address conversion:', error);
                return cborAddress;
            }
        }

        // Helper function to get any address from the wallet (CBOR or Bech32)
        async function getAnyAddress(walletAPI) {
            try {
                console.log('=== GETTING WALLET ADDRESS ===');
                console.log('Wallet API object:', walletAPI);

                // Method 1: Try getChangeAddress first
                try {
                    console.log('Attempting getChangeAddress...');
                    const address = await walletAPI.getChangeAddress();
                    console.log('getChangeAddress returned:', address);
                    console.log('Address type:', typeof address);
                    console.log('Address length:', address ? address.length : 'null');

                    if (address) {
                        // For CBOR-encoded addresses, convert to Bech32 first
                        if (address.startsWith('01') || address.startsWith('0x01')) {
                            console.log('CBOR address detected, converting to Bech32:', address);
                            const convertedAddress = await convertCborToBech32ViaAnvil(address);
                            console.log('Converted to Bech32:', convertedAddress);
                            return convertedAddress;
                        } else {
                            // For already Bech32 addresses, use as-is
                            console.log('Bech32 address detected, using as-is:', address);
                            return address;
                        }
                    }
                } catch (e) {
                    console.error('getChangeAddress failed:', e);
                }
                
                // Method 2: Try getUsedAddresses
                try {
                    const usedAddresses = await walletAPI.getUsedAddresses();
                    console.log('Method 2 - getUsedAddresses:', usedAddresses);
                    
                    if (usedAddresses && usedAddresses.length > 0) {
                        const address = usedAddresses[0];
                        if (address.startsWith('01') || address.startsWith('0x01')) {
                            console.log('Converting CBOR used address to Bech32:', address);
                            const convertedAddress = await convertCborToBech32ViaAnvil(address);
                            console.log('Converted used address:', convertedAddress);
                            return convertedAddress;
                        } else {
                            console.log('Using Bech32 used address as-is:', address);
                            return address;
                        }
                    }
                } catch (e) {
                    console.warn('getUsedAddresses failed:', e);
                }
                
                // Method 3: Try getUnusedAddresses
                try {
                    const unusedAddresses = await walletAPI.getUnusedAddresses();
                    console.log('Method 3 - getUnusedAddresses:', unusedAddresses);
                    
                    if (unusedAddresses && unusedAddresses.length > 0) {
                        const address = unusedAddresses[0];
                        if (address.startsWith('01') || address.startsWith('0x01')) {
                            console.log('Converting CBOR unused address to Bech32:', address);
                            const convertedAddress = await convertCborToBech32ViaAnvil(address);
                            console.log('Converted unused address:', convertedAddress);
                            return convertedAddress;
                        } else {
                            console.log('Using Bech32 unused address as-is:', address);
                            return address;
                        }
                    }
                } catch (e) {
                    console.warn('getUnusedAddresses failed:', e);
                }
                
                // Method 4: Try getRewardAddresses
                try {
                    const rewardAddresses = await walletAPI.getRewardAddresses();
                    console.log('Method 4 - getRewardAddresses:', rewardAddresses);
                    
                    if (rewardAddresses && rewardAddresses.length > 0) {
                        const address = rewardAddresses[0];
                        if (address.startsWith('01') || address.startsWith('0x01')) {
                            console.log('Converting CBOR reward address to Bech32:', address);
                            const convertedAddress = await convertCborToBech32ViaAnvil(address);
                            console.log('Converted reward address:', convertedAddress);
                            return convertedAddress;
                        } else {
                            console.log('Using Bech32 reward address as-is:', address);
                            return address;
                        }
                    }
                } catch (e) {
                    console.warn('getRewardAddresses failed:', e);
                }
                
                console.error('No address found through any method');
                return null;
                
            } catch (error) {
                console.error('Error getting address:', error);
                return null;
            }
        }

    // Initialize when DOM is loaded
    document.addEventListener('DOMContentLoaded', function() {
        console.log('MINT NOW script loaded and DOM ready');
        console.log('Looking for MINT NOW button...');
        const testBtn = document.getElementById('cardano-mint-now-btn');
        console.log('Button found at DOM ready:', testBtn);
        initializeNFTMint();
    });

    // Re-initialize when Bricks finishes rendering
    document.addEventListener('bricks:after:render', function() {
        console.log('Bricks render complete, re-initializing...');
        console.log('Looking for MINT NOW button after Bricks render...');
        const testBtn = document.getElementById('cardano-mint-now-btn');
        console.log('Button found after Bricks render:', testBtn);
        initializeNFTMint();
    });

    // Fallback: Re-initialize periodically for Bricks compatibility
    setTimeout(function() {
        console.log('Fallback timeout triggered - checking for button...');
        const testBtn = document.getElementById('cardano-mint-now-btn');
        console.log('Button found at timeout:', testBtn);
        initializeNFTMint();
    }, 2000);

    // Global event delegation - only set up once
    let eventDelegationSetup = false;
    
    function setupEventDelegation() {
        if (eventDelegationSetup) return;
        eventDelegationSetup = true;
        
        console.log('Setting up event delegation for MINT NOW button');
        
        // Use event delegation - listen on document for clicks on the button
        // This works even if Bricks replaces the DOM
        document.addEventListener('click', function(e) {
            console.log('Click detected on:', e.target, 'ID:', e.target.id, 'Classes:', e.target.className);
            
            if (e.target && e.target.id === 'cardano-mint-now-btn') {
                console.log('MINT NOW button clicked via delegation');
                e.preventDefault();
                openMintModal();
            }
            
            // Handle modal close button
            if (e.target && e.target.classList.contains('cardano-modal-close')) {
                console.log('Modal close button clicked via delegation');
                e.preventDefault();
                closeMintModal();
            }
            
            // Handle modal background click
            if (e.target && e.target.id === 'cardano-nft-mint-modal') {
                console.log('Modal background clicked via delegation');
                closeMintModal();
            }
        });
    }

    function initializeNFTMint() {
        console.log('initializeNFTMint() called');
        
        // Set up event delegation if not already done
        setupEventDelegation();

        // Modal event handling is now done via delegation above
        console.log('Modal event handling set up via delegation');

        // Initialize wallet connection
        console.log('Initializing wallet connection...');
        initializeWalletConnection();
    }

    function openMintModal() {
        const modal = document.getElementById('cardano-nft-mint-modal');
        if (modal) {
            modal.style.display = 'flex';
            // Allow modal to scroll but keep body scrollable too
            document.body.style.overflow = 'auto';
            nextMintStep(1);
        }
    }

    function closeMintModal() {
        console.log('closeMintModal() called');
        const modal = document.getElementById('cardano-nft-mint-modal');
        if (modal) {
            modal.style.display = 'none';
            document.body.style.overflow = 'auto'; // Restore scroll
            // Reset form
            const form = document.getElementById('cardano-nft-mint-form');
            if (form) {
                form.reset();
            }
            nextMintStep(1);
            console.log('MINT NOW modal closed successfully');
        } else {
            console.log('MINT NOW modal not found when trying to close');
        }
    }

    function nextMintStep(step) {
        // Hide all steps
        document.querySelectorAll('.mint-step').forEach(s => s.style.display = 'none');
        document.querySelectorAll('.mint-steps .step').forEach(s => s.classList.remove('active'));
        
        // Show current step
        const stepElement = document.getElementById('mint-step-' + step);
        if (stepElement) {
            stepElement.style.display = 'block';
        }
        
        const stepIndicator = document.querySelector('.mint-steps .step[data-step="' + step + '"]');
        if (stepIndicator) {
            stepIndicator.classList.add('active');
        }
    }

    function prevMintStep(step) {
        nextMintStep(step);
    }

    function initializeWalletConnection() {
        console.log('initializeWalletConnection() called');
        
        const connectBtn = document.getElementById('connect-wallet-btn');
        const proceedBtn = document.getElementById('proceed-to-confirm');
        const walletDisplay = document.getElementById('wallet-address-display');
        const walletAddress = document.getElementById('connected-wallet-address');
        const walletName = document.getElementById('connected-wallet-name');
        const walletNameDisplay = document.getElementById('wallet-name-display');
        const walletInput = document.getElementById('wallet-address');
        
        if (!connectBtn) {
            console.log('Connect wallet button not found, returning');
            return;
        }
        
        console.log('Connect wallet button found, setting up event listener');
        
        // Initialize CIP-30 wallet system
        initializeCIP30Wallet();
    
        function initializeCIP30Wallet() {
            console.log('initializeCIP30Wallet() called');
            console.log('window.cardanoMint exists:', !!window.cardanoMint);
            if (window.cardanoMint) {
                console.log('cardanoMint.debug:', window.cardanoMint.debug);
                console.log('cardanoMint object:', window.cardanoMint);
            }
            if (window.cardanoMint && window.cardanoMint.debug) {
                console.log('CIP-30 wallet system initialized for minting');
            } else {
                console.log('Debug mode not enabled, but CIP-30 system is ready');
            }
        }
    
        // Detect available Cardano wallets using CIP-30 (universal approach)
        function detectAvailableWallets() {
            const wallets = [];
            
            console.log('Checking for Cardano wallets...');
            console.log('window.cardano exists:', !!window.cardano);
            
            if (window.cardano) {
                // Check for CardanoPress-supported wallet APIs
                const walletNames = [
                    'eternl', 'yoroi', 'typhon', 'gero', 'nufi', 'lace', 'begin', 'vespr'
                ];
                
                console.log('Checking known wallet names:', walletNames);
                
                walletNames.forEach(walletName => {
                    if (window.cardano[walletName] && window.cardano[walletName].enable) {
                        console.log('Valid wallet found:', walletName);
                        wallets.push({
                            name: walletName,
                            displayName: getWalletDisplayName(walletName),
                            api: window.cardano[walletName]
                        });
                    } else {
                        console.log('Wallet not available:', walletName);
                    }
                });
            } else {
                console.log('window.cardano is not available');
            }
            
            console.log('Total wallets found:', wallets.length);
            
            return wallets;
        }
        
        // Get display name for wallet (CardanoPress-compatible)
        function getWalletDisplayName(walletName) {
            const displayNames = {
                'eternl': 'Eternl',
                'yoroi': 'Yoroi',
                'typhon': 'Typhon',
                'gero': 'GeroWallet',
                'nufi': 'NuFi',
                'lace': 'Lace',
                'begin': 'Begin',
                'vespr': 'VESPR'
            };
            
            return displayNames[walletName] || walletName;
        }
        
        // Helper function to connect to CardanoPress wallet
        async function connectToCardanoPressWallet(cardanoPressWallet) {
            try {
                const walletAPI = await window.cardano[cardanoPressWallet.name].enable();
                const walletAddressValue = await getAnyAddress(walletAPI);
                
                if (!walletAddressValue) {
                    throw new Error('Could not get wallet address');
                }
                
                // Create display address for UI (human-readable)
                let displayAddress = walletAddressValue;
                if (walletAddressValue.startsWith('01') || walletAddressValue.startsWith('0x01')) {
                    // For CBOR addresses, convert to Bech32 format for display
                    const cleanAddress = walletAddressValue.startsWith('0x') ? walletAddressValue.slice(2) : walletAddressValue;
                    displayAddress = `addr1${cleanAddress}`;
                }
                
                // Store wallet information
                mintWallet = {
                    name: cardanoPressWallet.name,
                    displayName: cardanoPressWallet.displayName,
                    changeAddress: walletAddressValue, // Raw address for backend
                    displayAddress: displayAddress,    // Human-readable for UI
                    api: walletAPI,
                    isCardanoPress: true,
                    signTx: async (tx) => {
                        return await walletAPI.signTx(tx, true);
                    }
                };
                
                // Save wallet choice for future use
                saveWalletChoice(mintWallet);
                
                if (window.cardanoMint && window.cardanoMint.debug) {
                    console.log('CardanoPress wallet connected:', mintWallet.displayAddress);
                }

                // Update UI - show wallet name
                if (walletName) {
                    walletName.textContent = cardanoPressWallet.displayName;
                }
                if (walletNameDisplay) {
                    walletNameDisplay.style.display = 'block';
                }

                // Update UI - show wallet address
                if (walletAddress) {
                    walletAddress.textContent = mintWallet.displayAddress;
                }
                if (walletDisplay) {
                    walletDisplay.style.display = 'block';
                }
                if (walletInput) {
                    walletInput.value = mintWallet.displayAddress;
                }

                connectBtn.textContent = 'Using Connected Wallet ✓';
                connectBtn.style.backgroundColor = '#28a745';

                if (proceedBtn) proceedBtn.style.display = 'block';
                
            } catch (error) {
                throw new Error('Failed to connect to CardanoPress wallet: ' + error.message);
            }
        }
        
        // Helper function to connect to regular wallet
        async function connectToRegularWallet(selectedWallet) {
            const walletAPI = await selectedWallet.api.enable();
            const walletAddressValue = await getAnyAddress(walletAPI);

            if (!walletAddressValue) {
                throw new Error('Could not get wallet address');
            }

            // Create display address for UI - format for readability
            let displayAddress = walletAddressValue;

            // If CBOR hex address, format it nicely for display (first 20 chars...last 20 chars)
            if (walletAddressValue.startsWith('01') && walletAddressValue.length > 50) {
                displayAddress = walletAddressValue.substring(0, 20) + '...' + walletAddressValue.substring(walletAddressValue.length - 20);
            } else if (walletAddressValue.startsWith('addr1') && walletAddressValue.length > 50) {
                // If already Bech32, also shorten for display
                displayAddress = walletAddressValue.substring(0, 20) + '...' + walletAddressValue.substring(walletAddressValue.length - 20);
            }

            // Store wallet information
            mintWallet = {
                name: selectedWallet.name,
                displayName: selectedWallet.displayName,
                changeAddress: walletAddressValue, // Raw address for backend
                displayAddress: displayAddress,    // Shortened for UI
                fullAddress: walletAddressValue,   // Full address stored
                api: walletAPI,
                isCardanoPress: false,
                signTx: async (tx) => {
                    return await walletAPI.signTx(tx, true);
                }
            };

            // Clear any cached wallet from different wallet provider
            localStorage.removeItem('cardano_mint_wallet');
            sessionStorage.removeItem('cardano_mint_wallet');
            localStorage.removeItem('cardanopress_wallet');
            sessionStorage.removeItem('cardanopress_wallet');

            // Save current wallet choice
            saveWalletChoice(mintWallet);

            console.log('Wallet connected:', selectedWallet.displayName);
            console.log('Address (shortened):', displayAddress);
            console.log('Full address:', walletAddressValue);

            // Update UI - show wallet name
            if (walletName) {
                walletName.textContent = selectedWallet.displayName;
            }
            if (walletNameDisplay) {
                walletNameDisplay.style.display = 'block';
            }

            // Update UI - show wallet address
            if (walletAddress) {
                walletAddress.textContent = displayAddress;
            }
            if (walletDisplay) {
                walletDisplay.style.display = 'block';
            }
            if (walletInput) {
                walletInput.value = walletAddressValue; // Store full address in hidden field
            }

            connectBtn.textContent = 'Wallet Connected ✓';
            connectBtn.style.backgroundColor = '#28a745';

            if (proceedBtn) proceedBtn.style.display = 'block';
        }
        
        // Helper function to save wallet choice for persistence
        function saveWalletChoice(wallet) {
            try {
                const walletData = {
                    name: wallet.name,
                    displayName: wallet.displayName,
                    changeAddress: wallet.changeAddress,
                    isCardanoPress: wallet.isCardanoPress || false,
                    timestamp: Date.now()
                };
                
                // Save to localStorage for persistence across sessions
                localStorage.setItem('cardano_mint_wallet', JSON.stringify(walletData));
                
                // Also save to sessionStorage for current session
                sessionStorage.setItem('cardano_mint_wallet', JSON.stringify(walletData));
                
                if (window.cardanoMint && window.cardanoMint.debug) {
                    console.log('Wallet choice saved:', walletData);
                }
            } catch (error) {
                console.warn('Failed to save wallet choice:', error);
            }
        }
        
        // Helper function to load saved wallet choice
        function loadSavedWalletChoice() {
            try {
                // Try localStorage first
                const savedWallet = localStorage.getItem('cardano_mint_wallet');
                if (savedWallet) {
                    const walletData = JSON.parse(savedWallet);
                    // Check if wallet is still available
                    if (window.cardano && window.cardano[walletData.name] && window.cardano[walletData.name].enable) {
                        return walletData;
                    }
                }
                
                // Try sessionStorage as fallback
                const sessionWallet = sessionStorage.getItem('cardano_mint_wallet');
                if (sessionWallet) {
                    const walletData = JSON.parse(sessionWallet);
                    if (window.cardano && window.cardano[walletData.name] && window.cardano[walletData.name].enable) {
                        return walletData;
                    }
                }
                
                return null;
            } catch (error) {
                console.warn('Failed to load saved wallet choice:', error);
                return null;
            }
        }
    
        // Check for CardanoPress wallet on page load
        const cardanoPressWallet = detectCardanoPressWallet();
        if (cardanoPressWallet) {
            console.log('CardanoPress wallet detected, auto-connecting...');
            connectBtn.textContent = 'Using Connected Wallet...';
            connectBtn.disabled = true;
            
            // Auto-connect to CardanoPress wallet
            connectToCardanoPressWallet(cardanoPressWallet);
        }
        
        // Real wallet connection using CIP-30 with CardanoPress integration
        connectBtn.addEventListener('click', async function() {
        console.log('Wallet confirmation button clicked!');
        try {
            connectBtn.textContent = 'Confirming...';
            connectBtn.disabled = true;

            // Detect available wallets
            const availableWallets = detectAvailableWallets();
            console.log('Available wallets:', availableWallets);

            if (availableWallets.length === 0) {
                throw new Error('No Cardano wallets found. Please install a wallet extension.');
            }

            // Check CardanoPress for connected wallet using multiple methods
            let selectedWallet = null;
            let walletAPI = null;

            // Method 1: Check window.cardanoPress global object
            let cardanoPressWalletName = null;
            if (window.cardanoPress && window.cardanoPress.wallet && window.cardanoPress.wallet.name) {
                cardanoPressWalletName = window.cardanoPress.wallet.name.toLowerCase();
                console.log('CardanoPress connected wallet from global object:', cardanoPressWalletName);
            }

            // Method 2: Check localStorage (fallback)
            if (!cardanoPressWalletName) {
                const storedName = localStorage.getItem('_x_connectedExtension');
                if (storedName) {
                    cardanoPressWalletName = storedName.toLowerCase();
                    console.log('CardanoPress connected wallet from localStorage:', cardanoPressWalletName);
                }
            }

            if (cardanoPressWalletName) {
                // Find the wallet that matches CardanoPress connection (case-insensitive)
                const matchingWallet = availableWallets.find(w => w.name.toLowerCase() === cardanoPressWalletName);
                if (matchingWallet) {
                    console.log('Found matching wallet:', matchingWallet.displayName);
                    selectedWallet = matchingWallet;
                    try {
                        walletAPI = await matchingWallet.api.enable();
                        console.log('Successfully enabled CardanoPress wallet');
                    } catch (e) {
                        console.error('Failed to enable CardanoPress wallet:', e);
                        selectedWallet = null;
                    }
                } else {
                    console.warn('Could not find wallet matching:', cardanoPressWalletName);
                }
            }

            // If no wallet from CardanoPress, try isConnected()
            if (!selectedWallet) {
                for (const wallet of availableWallets) {
                    try {
                        console.log('Checking if', wallet.displayName, 'is already connected...');
                        if (wallet.api.isConnected && await wallet.api.isConnected()) {
                            console.log(wallet.displayName, 'is CONNECTED!');
                            selectedWallet = wallet;
                            walletAPI = await wallet.api.enable();
                            break;
                        }
                    } catch (e) {
                        console.log(wallet.displayName, 'not connected:', e.message);
                    }
                }
            }

            // If no wallet is actively connected, use the first available
            if (!selectedWallet) {
                console.log('No wallet actively connected, using first available');
                selectedWallet = availableWallets[0];
                walletAPI = await selectedWallet.api.enable();
            }

            console.log('Using wallet:', selectedWallet.displayName);

            // Get the address from CardanoPress (same as shown on homepage)
            console.log('Fetching address from CardanoPress...');
            let changeAddress = null;

            try {
                const formData = new FormData();
                formData.append('action', 'cardano_get_connected_wallet');
                formData.append('nonce', cardanoMint.nonce);

                const response = await fetch(cardanoMint.ajaxurl, {
                    method: 'POST',
                    body: formData
                });

                const result = await response.json();
                if (result.success && result.data.address) {
                    changeAddress = result.data.address;
                    console.log('Got address from CardanoPress:', changeAddress);
                }
            } catch (e) {
                console.warn('Failed to get address from CardanoPress:', e);
            }

            // Fallback: get from wallet API
            if (!changeAddress) {
                console.log('Fallback: Fetching address from wallet API...');
                changeAddress = await getAnyAddress(walletAPI);
                console.log('Got address from wallet API:', changeAddress);
            }

            if (!changeAddress) {
                throw new Error('Could not get wallet address');
            }

            // Format address for display
            let displayAddr = changeAddress;
            if (changeAddress && changeAddress.length > 50) {
                displayAddr = changeAddress.substring(0, 20) + '...' + changeAddress.substring(changeAddress.length - 20);
            }

            // Update UI - show wallet name
            if (walletName) {
                walletName.textContent = selectedWallet.displayName;
            }
            if (walletNameDisplay) {
                walletNameDisplay.style.display = 'block';
            }

            // Update UI - show wallet address
            if (walletAddress) {
                walletAddress.textContent = displayAddr;
            }
            if (walletDisplay) {
                walletDisplay.style.display = 'block';
            }
            if (walletInput) {
                walletInput.value = changeAddress;
            }

            // Store wallet for later use
            mintWallet = {
                name: selectedWallet.name,
                displayName: selectedWallet.displayName,
                changeAddress: changeAddress,
                displayAddress: displayAddr,
                fullAddress: changeAddress,
                api: walletAPI,
                isCardanoPress: false,
                signTx: async (tx) => {
                    return await walletAPI.signTx(tx, true);
                }
            };

            console.log('Wallet connected successfully:', mintWallet);

            connectBtn.textContent = 'Wallet Confirmed ✓';
            connectBtn.style.backgroundColor = '#28a745';

            if (proceedBtn) proceedBtn.style.display = 'block';
            
        } catch (error) {
            if (window.cardanoMint && cardanoMint.debug) {
                console.error('Mint wallet confirmation failed:', error);
            }
            alert('Wallet confirmation failed: ' + error.message);
            if (connectBtn) {
                connectBtn.textContent = 'Confirm Wallet';
                connectBtn.disabled = false;
            }
        }
    });
    
        if (proceedBtn) {
            proceedBtn.addEventListener('click', function() {
                nextMintStep(2);
                const confirmWalletAddress = document.getElementById('confirm-wallet-address');
                if (confirmWalletAddress && walletInput) {
                    confirmWalletAddress.textContent = walletInput.value;
                }
            });
        }
    
        // Real mint confirmation with payment processing
        const confirmMintBtn = document.getElementById('confirm-mint-btn');
        if (confirmMintBtn) {
            confirmMintBtn.addEventListener('click', async function() {
                if (isMintProcessing) return;
                
                const confirmBtn = document.getElementById('confirm-mint-btn');
                confirmBtn.textContent = 'Processing Mint...';
                confirmBtn.disabled = true;
                isMintProcessing = true;
                
                try {
                    if (!mintWallet || !mintWallet.changeAddress) {
                        throw new Error('Wallet not confirmed');
                    }
                    
                    // CRITICAL FIX: Get merchant address from the MINT NOW button's data attribute
                    const mintButton = document.getElementById('cardano-mint-now-btn');
                    const merchantAddress = mintButton?.dataset.merchantAddress || '';
                    
                    // Get other mint data from hidden fields/data attributes
                    const policyId = document.getElementById('policy-id')?.value ||
                                   mintButton?.dataset.policyId || '';

                    // DEBUG: Check all price sources
                    const hiddenPriceField = document.getElementById('nft-price');
                    const hiddenPriceValue = hiddenPriceField?.value;
                    const buttonPriceValue = mintButton?.dataset.nftPrice;
                    console.log('DEBUG Price Sources:');
                    console.log('  - Hidden field value:', hiddenPriceValue);
                    console.log('  - Button data-nft-price:', buttonPriceValue);

                    const mintPrice = parseFloat(hiddenPriceValue || buttonPriceValue || '0');

                    console.log('=== MINT TRANSACTION DATA ===');
                    console.log('Merchant Address:', merchantAddress);
                    console.log('Customer Address:', mintWallet.changeAddress);
                    console.log('Price (USD):', mintPrice);
                    console.log('Policy ID:', policyId);
                    console.log('============================');
                    
                    // Validate merchant address exists
                    if (!merchantAddress || merchantAddress.trim() === '') {
                        throw new Error('Merchant address not configured. Please contact the site administrator.');
                    }
                    
                    // Step 1: Build mint transaction
                    console.log('Building mint transaction...');
                    console.log('Calling buildMintTransaction with price:', mintPrice);
                    const buildData = await buildMintTransaction(
                        merchantAddress,
                        mintWallet.changeAddress,
                        mintPrice,
                        policyId
                    );
                    
                    if (!buildData.complete) {
                        throw new Error('Failed to build mint transaction');
                    }
                    
                    // Step 2: Sign transaction
                    console.log('Please sign the transaction in your wallet...');
                    let signature;
                    try {
                        signature = await mintWallet.signTx(buildData.complete);
                    } catch (signError) {
                        // Check if this is a user decline error after successful submission
                        if (signError.name === 'TxSignError' && signError.code === 2) {
                            console.warn('Transaction signing declined by user, but transaction may have already been submitted');
                            signature = null;
                        } else {
                            throw signError;
                        }
                    }
                    
                    // Step 3: Submit transaction (only if we have a signature)
                    if (signature) {
                        console.log('Submitting mint transaction...');

                        // Build signatures array: Anvil's policy witness + user's wallet signature
                        const signatures = [];

                        // Add Anvil's policy script witness if available (REQUIRED for minting!)
                        if (buildData.witnessSet) {
                            signatures.push(buildData.witnessSet);
                            console.log('Added policy script witness from Anvil');
                        }

                        // Add user's wallet signature
                        signatures.push(signature);
                        console.log('Added user wallet signature');
                        console.log('Total signatures:', signatures.length);

                        const submitResult = await submitMintTransaction(
                            buildData.complete,
                            signatures,
                            policyId,
                            mintWallet.changeAddress
                        );
                        
                        if (!submitResult.txHash) {
                            throw new Error('Mint transaction submission failed');
                        }
                        
                        // Success!
                        const txHashElement = document.getElementById('mint-tx-hash');
                        if (txHashElement) {
                            txHashElement.textContent = submitResult.txHash;
                        }
                        nextMintStep(3);
                    } else {
                        // If no signature but no error, assume transaction was already processed
                        console.log('Transaction may have been processed by wallet directly');
                        const txHashElement = document.getElementById('mint-tx-hash');
                        if (txHashElement) {
                            txHashElement.textContent = 'Processed by wallet';
                        }
                        nextMintStep(3);
                    }
                    
                } catch (error) {
                    console.error('Mint failed:', error);
                    alert('Mint failed: ' + error.message);
                } finally {
                    if (confirmBtn) {
                        confirmBtn.textContent = 'CONFIRM MINT';
                        confirmBtn.disabled = false;
                    }
                    isMintProcessing = false;
                }
            });
        }
    }

    // Build mint transaction
    async function buildMintTransaction(merchantAddress, customerAddress, usdPrice, policyId) {
        if (!window.cardanoMint) {
            throw new Error('Cardano Mint configuration not found');
        }
        
        // CRITICAL FIX: Validate ALL inputs before making API call
        if (!merchantAddress || merchantAddress.trim() === '') {
            console.error('Merchant address is missing or empty');
            throw new Error('Merchant address not configured. Please contact the site administrator.');
        }
        
        if (!customerAddress || customerAddress.trim() === '') {
            console.error('Customer address is missing or empty');
            throw new Error('Wallet address is required');
        }
        
        if (!usdPrice || usdPrice <= 0) {
            console.error('Invalid USD price:', usdPrice);
            throw new Error('Invalid price');
        }
        
        if (!policyId || policyId.trim() === '') {
            console.error('Policy ID is missing or empty');
            throw new Error('Policy ID is required');
        }
        
        console.log('✅ All validations passed! Building transaction...');
        console.log('Merchant:', merchantAddress);
        console.log('Customer:', customerAddress);
        console.log('Price:', usdPrice);
        console.log('Policy:', policyId);

        // Get asset ID from the button data attribute
        const mintButton = document.getElementById('cardano-mint-now-btn');
        const assetId = mintButton ? mintButton.getAttribute('data-mint-id') : '';
        console.log('Asset ID:', assetId);

        const formData = new FormData();
        formData.append('action', 'cardano_build_mint_transaction');
        formData.append('nonce', cardanoMint.nonce);
        formData.append('merchant_address', merchantAddress);
        formData.append('customer_address', customerAddress);
        formData.append('usd_price', usdPrice);
        formData.append('policy_id', policyId);
        formData.append('asset_id', assetId);

        // DEBUG: Log FormData contents
        console.log('=== FORM DATA BEING SENT ===');
        for (let pair of formData.entries()) {
            console.log(pair[0] + ': ' + pair[1]);
        }
        console.log('============================');

        const response = await fetch(cardanoMint.ajaxurl, {
            method: 'POST',
            body: formData
        });
        
        const result = await response.json();

        // DEBUG: Log response from PHP
        console.log('=== PHP RESPONSE ===');
        console.log('Success:', result.success);
        if (result.data && result.data.debug_price_info) {
            console.log('PHP received usd_price:', result.data.debug_price_info.usd_price_received);
            console.log('Raw POST price:', result.data.debug_price_info.raw_post_price);
        }
        console.log('====================');

        if (!result.success) {
            throw new Error(result.data?.message || 'Failed to build mint transaction');
        }

        return result.data;
    }

    // Submit mint transaction
    async function submitMintTransaction(transaction, signatures, policyId, walletAddress) {
        if (!window.cardanoMint) {
            throw new Error('Cardano Mint configuration not found');
        }
        
        // Get asset ID from the button data attribute
        const mintButton = document.getElementById('cardano-mint-now-btn');
        const assetId = mintButton ? mintButton.getAttribute('data-mint-id') : '';

        const formData = new FormData();
        formData.append('action', 'cardano_submit_mint_transaction');
        formData.append('nonce', cardanoMint.nonce);
        formData.append('transaction', transaction);
        formData.append('signatures', JSON.stringify(signatures));
        formData.append('policy_id', policyId);
        formData.append('wallet_address', walletAddress);
        formData.append('asset_id', assetId);
        
        const response = await fetch(cardanoMint.ajaxurl, {
            method: 'POST',
            body: formData
        });
        
        const result = await response.json();
        
        if (!result.success) {
            throw new Error(result.data?.message || 'Failed to submit mint transaction');
        }
        
        return result.data;
    }

    // Make functions globally available
    window.nextMintStep = nextMintStep;
    window.prevMintStep = prevMintStep;
    window.closeMintModal = closeMintModal;

})(); // End of IIFE