// Cardano Checkout JavaScript with Real Payment Processing

// Global variables
let currentStep = 1;
let productData = {};
let adaPrice = 1.0;
let wallet = null;
let isProcessing = false;

// Initialize when DOM is loaded
document.addEventListener('DOMContentLoaded', function() {
    initializeCheckout();
    initializeCIP30Wallet();
});

function initializeCheckout() {
    // Get BUY NOW button
    const buyNowBtn = document.getElementById('cardano-buy-now-btn');
    if (buyNowBtn) {
        buyNowBtn.addEventListener('click', openCheckoutModal);
    }

    // Get modal close button (specific to BUY NOW modal)
    const modalClose = document.querySelector('#cardano-checkout-modal .cardano-modal-close');
    if (modalClose) {
        modalClose.addEventListener('click', closeCheckoutModal);
    }

    // Close modal when clicking outside
    const modal = document.getElementById('cardano-checkout-modal');
    if (modal) {
        modal.addEventListener('click', function(e) {
            if (e.target === modal) {
                closeCheckoutModal();
            }
        });
    }

    // Handle form submission - now requires payment first
    const checkoutForm = document.getElementById('cardano-checkout-form');
    if (checkoutForm) {
        checkoutForm.addEventListener('submit', handleOrderSubmissionWithPayment);
    }

    // Load ADA price
    loadAdaPrice();
    
    // Initialize real-time validation
    initializeRealTimeValidation();
    
    // Initialize field styling
    initializeFieldStyling();
}

/**
 * Initialize CIP-30 wallet system
 */
function initializeCIP30Wallet() {
    console.log('CIP-30 wallet system initialized');
    // No initialization needed for CIP-30 - it's always available
}

/**
 * Detect CardanoPress connected wallet
 */
function detectCardanoPressWallet() {
    try {
        // Check for CardanoPress wallet connection
        if (window.cardanopress && window.cardanopress.wallet) {
            const cpWallet = window.cardanopress.wallet;
            if (cpWallet.name && cpWallet.address) {
                console.log('CardanoPress wallet detected:', cpWallet);
                return {
                    name: cpWallet.name,
                    displayName: getWalletDisplayName(cpWallet.name),
                    address: cpWallet.address,
                    isCardanoPress: true
                };
            }
        }
        
        // Check localStorage for CardanoPress wallet info
        const storedWallet = localStorage.getItem('cardanopress_wallet');
        if (storedWallet) {
            try {
                const walletData = JSON.parse(storedWallet);
                if (walletData.name && walletData.address) {
                    console.log('CardanoPress wallet from localStorage:', walletData);
                    return {
                        name: walletData.name,
                        displayName: getWalletDisplayName(walletData.name),
                        address: walletData.address,
                        isCardanoPress: true
                    };
                }
            } catch (e) {
                console.warn('Failed to parse stored CardanoPress wallet:', e);
            }
        }
        
        // Check sessionStorage as fallback
        const sessionWallet = sessionStorage.getItem('cardanopress_wallet');
        if (sessionWallet) {
            try {
                const walletData = JSON.parse(sessionWallet);
                if (walletData.name && walletData.address) {
                    console.log('CardanoPress wallet from sessionStorage:', walletData);
                    return {
                        name: walletData.name,
                        displayName: getWalletDisplayName(walletData.name),
                        address: walletData.address,
                        isCardanoPress: true
                    };
                }
            } catch (e) {
                console.warn('Failed to parse session CardanoPress wallet:', e);
            }
        }
        
        return null;
    } catch (error) {
        console.warn('Error detecting CardanoPress wallet:', error);
        return null;
    }
}

/**
 * Detect available Cardano wallets using CIP-30
 */
function detectAvailableWallets() {
    const wallets = [];
    
    if (window.cardano) {
        // Check for CardanoPress-supported wallet APIs
        const walletNames = [
            'eternl', 'yoroi', 'typhon', 'gero', 'nufi', 'lace', 'begin', 'vespr'
        ];
        
        walletNames.forEach(walletName => {
            if (window.cardano[walletName] && window.cardano[walletName].enable) {
                wallets.push({
                    name: walletName,
                    displayName: getWalletDisplayName(walletName),
                    api: window.cardano[walletName]
                });
            }
        });
    }
    
    return wallets;
}

/**
 * Get display name for wallet
 */
function getWalletDisplayName(walletName) {
    const displayNames = {
        'eternl': 'Eternl',
        'yoroi': 'Yoroi',
        'typhon': 'Typhon',
        'gero': 'GeroWallet',
        'nufi': 'NuFi',
        'lace': 'Lace',
        'begin': 'Begin',
        'vespr': 'VESPR'
    };
    return displayNames[walletName] || walletName;
}

/**
 * Save wallet choice for persistence
 */
function saveWalletChoice(wallet) {
    try {
        const walletData = {
            name: wallet.name,
            displayName: wallet.displayName,
            changeAddress: wallet.changeAddress,
            isCardanoPress: wallet.isCardanoPress || false,
            timestamp: Date.now()
        };
        
        // Save to localStorage for persistence across sessions
        localStorage.setItem('cardano_pay_wallet', JSON.stringify(walletData));
        
        // Also save to sessionStorage for current session
        sessionStorage.setItem('cardano_pay_wallet', JSON.stringify(walletData));
        
        if (cardanoCheckout.debug) {
            console.log('Wallet choice saved:', walletData);
        }
    } catch (error) {
        console.warn('Failed to save wallet choice:', error);
    }
}

/**
 * Load saved wallet choice
 */
function loadSavedWalletChoice() {
    try {
        // Try localStorage first
        const savedWallet = localStorage.getItem('cardano_pay_wallet');
        if (savedWallet) {
            const walletData = JSON.parse(savedWallet);
            // Check if wallet is still available
            if (window.cardano && window.cardano[walletData.name] && window.cardano[walletData.name].enable) {
                return walletData;
            }
        }
        
        // Try sessionStorage as fallback
        const sessionWallet = sessionStorage.getItem('cardano_pay_wallet');
        if (sessionWallet) {
            const walletData = JSON.parse(sessionWallet);
            if (window.cardano && window.cardano[walletData.name] && window.cardano[walletData.name].enable) {
                return walletData;
            }
        }
        
        return null;
    } catch (error) {
        console.warn('Failed to load saved wallet choice:', error);
        return null;
    }
}

function openCheckoutModal() {
    const btn = document.getElementById('cardano-buy-now-btn');
    if (!btn) return;

    // Check if API is configured
    if (!cardanoCheckout.apiConfigured) {
        alert('Cardano Pay is not properly configured. Please contact the site administrator to set up the Anvil API key.');
        return;
    }
    
    if (!cardanoCheckout.merchantConfigured) {
        alert('Cardano Pay merchant address is not configured. Please contact the site administrator.');
        return;
    }

    // Store product data
    productData = {
        productId: btn.dataset.productId,
        productName: btn.dataset.productName,
        productPrice: parseFloat(btn.dataset.productPrice),
        productImage: btn.dataset.productImage,
        productDescription: btn.dataset.productDescription,
        quantityLimit: parseInt(btn.dataset.quantityLimit),
        merchantAddress: btn.dataset.merchantAddress
    };

    // Populate quantity dropdown
    populateQuantityDropdown();

    // Show modal
    const modal = document.getElementById('cardano-checkout-modal');
    if (modal) {
        modal.style.display = 'flex';
        document.body.style.overflow = 'hidden';
    }

    // Reset to step 1
    showStep(1);
}

function closeCheckoutModal() {
    const modal = document.getElementById('cardano-checkout-modal');
    if (modal) {
        modal.style.display = 'none';
        document.body.style.overflow = 'auto';
    }
    
    // Reset form
    resetForm();
}

function showStep(stepNumber) {
    // Hide all steps
    const steps = document.querySelectorAll('.checkout-step');
    steps.forEach(step => step.style.display = 'none');

    // Show current step
    const currentStepElement = document.getElementById('step-' + stepNumber);
    if (currentStepElement) {
        currentStepElement.style.display = 'block';
    }

    // Update step indicators
    const stepIndicators = document.querySelectorAll('.step');
    stepIndicators.forEach((indicator, index) => {
        indicator.classList.remove('active', 'completed');
        if (index + 1 < stepNumber) {
            indicator.classList.add('completed');
        } else if (index + 1 === stepNumber) {
            indicator.classList.add('active');
        }
    });

    currentStep = stepNumber;

    // If step 3, populate review data
    if (stepNumber === 3) {
        populateOrderReview();
    }
}

function nextStep(stepNumber) {
    if (validateCurrentStep()) {
        showStep(stepNumber);
    }
}

function prevStep(stepNumber) {
    showStep(stepNumber);
}

function validateCurrentStep() {
    const currentStepElement = document.getElementById('step-' + currentStep);
    if (!currentStepElement) return false;

    const requiredFields = currentStepElement.querySelectorAll('input[required], select[required], textarea[required]');
    let isValid = true;

    requiredFields.forEach(field => {
        if (!field.value.trim()) {
            field.style.borderColor = '#dc2626';
            isValid = false;
        } else {
            field.style.borderColor = '#e5e7eb';
        }
    });

    if (!isValid) {
        alert('Please fill in all required fields.');
    }

    return isValid;
}

function populateQuantityDropdown() {
    const quantitySelect = document.getElementById('review_quantity');
    if (!quantitySelect || !productData.quantityLimit) return;

    quantitySelect.innerHTML = '';
    for (let i = 1; i <= productData.quantityLimit; i++) {
        const option = document.createElement('option');
        option.value = i;
        option.textContent = i;
        quantitySelect.appendChild(option);
    }
}

function populateOrderReview() {
    // Populate product details
    document.getElementById('review_product_name').textContent = productData.productName || 'Product';
    document.getElementById('review_product_description').textContent = productData.productDescription || '';
    if (productData.productImage) {
        document.getElementById('review_product_image').src = productData.productImage;
        document.getElementById('review_product_image').alt = productData.productName || 'Product';
    }
    
    // Populate customer details
    const firstName = document.getElementById('first_name').value;
    const lastName = document.getElementById('last_name').value;
    const email = document.getElementById('email').value;
    const phone = document.getElementById('phone').value;
    
    document.getElementById('review_customer_name').textContent = firstName + ' ' + lastName;
    document.getElementById('review_email').textContent = email;
    document.getElementById('review_phone').textContent = phone;
    
    // Populate shipping details
    const address1 = document.getElementById('shipping_address_1').value;
    const address2 = document.getElementById('shipping_address_2').value;
    const city = document.getElementById('city').value;
    const state = document.getElementById('state_province').value;
    const postal = document.getElementById('postal_code').value;
    const country = document.getElementById('country').value;
    
    let shippingAddress = address1;
    if (address2) shippingAddress += ', ' + address2;
    shippingAddress += ', ' + city + ', ' + state + ' ' + postal + ', ' + country;
    
    document.getElementById('review_shipping_address').textContent = shippingAddress;
    
    // Update pricing
    updateOrderTotal();
    
    // Set hidden fields
    document.getElementById('order_id').value = 'ORD_' + Date.now();
    document.getElementById('order_date').value = new Date().toISOString();
    document.getElementById('product_id').value = productData.productId;
    document.getElementById('merchant_address').value = productData.merchantAddress;
}

function updateOrderTotal() {
    const quantity = parseInt(document.getElementById('review_quantity').value) || 1;
    const unitPrice = productData.productPrice;
    const totalUSD = unitPrice * quantity;
    const totalADA = totalUSD / adaPrice;
    
    document.getElementById('order_total_usd').textContent = '$' + totalUSD.toFixed(2) + ' USD';
    document.getElementById('order_total_ada').textContent = '(' + totalADA.toFixed(4) + ' ADA)';
}

function loadAdaPrice() {
    // Try to get cached price first
    const cachedData = localStorage.getItem('cardano_ada_price_data');
    if (cachedData) {
        const { price, timestamp } = JSON.parse(cachedData);
        const now = Date.now();
        const cacheAge = now - timestamp;
        const cacheExpiry = 5 * 60 * 1000; // 5 minutes
        
        if (cacheAge < cacheExpiry) {
            adaPrice = price;
            console.log('Using cached ADA price:', adaPrice);
            return;
        }
    }
    
    // Fetch current price
    console.log('Fetching fresh ADA price...');
    fetch('https://api.coingecko.com/api/v3/simple/price?ids=cardano&vs_currencies=usd')
        .then(response => response.json())
        .then(data => {
            if (data.cardano && data.cardano.usd) {
                adaPrice = data.cardano.usd;
                const cacheData = {
                    price: adaPrice,
                    timestamp: Date.now()
                };
                localStorage.setItem('cardano_ada_price_data', JSON.stringify(cacheData));
                console.log('Updated ADA price:', adaPrice);
            }
        })
        .catch(error => {
            console.log('Could not fetch ADA price, using cached or default:', error);
            // Try to use any cached price as fallback
            if (cachedData) {
                const { price } = JSON.parse(cachedData);
                adaPrice = price;
            }
        });
}

/**
 * Handle order submission with real payment processing
 */
async function handleOrderSubmissionWithPayment(e) {
    e.preventDefault();
    
    if (isProcessing) return;
    
    // Show loading state
    const submitBtn = document.querySelector('.btn-place-order');
    if (submitBtn) {
        submitBtn.textContent = 'Processing Payment...';
        submitBtn.disabled = true;
    }
    
    try {
        isProcessing = true;
        
        // Step 1: Connect wallet if not already connected
        if (!wallet || !wallet.changeAddress) {
            showStatus('Connecting to wallet...');
            await connectWallet();
        }
        
        // Step 2: Build transaction
        showStatus('Building transaction...');
        const buildData = await buildPaymentTransaction();
        
        if (!buildData.transaction) {
            throw new Error('Failed to build transaction');
        }
        
        // Log transaction details for debugging
        if (cardanoCheckout.debug) {
            console.log('Built transaction:', buildData.transaction);
        }
        
        // Step 3: Sign transaction
        showStatus('Please sign the transaction in your wallet...');
        
        // Sign the real transaction with the wallet
        let signature;
        try {
            // Pass the raw transaction string (complete field) to wallet.signTx
            const rawTransaction = buildData.transaction.complete || buildData.transaction;
            if (cardanoCheckout.debug) {
            console.log('Signing raw transaction:', rawTransaction.substring(0, 100) + '...');
        }
            signature = await wallet.signTx(rawTransaction);
            if (cardanoCheckout.debug) {
                console.log('Transaction signed successfully');
            }
        } catch (signError) {
            console.error('Wallet signing failed:', signError);
            throw new Error('Transaction signing failed: ' + signError.message);
        }
        
        // Step 4: Submit transaction to blockchain
        showStatus('Submitting transaction to blockchain...');
        const submitResult = await submitPaymentTransaction(buildData.transaction, [signature]);
        
        if (cardanoCheckout.debug) {
            console.log('Submit result:', submitResult);
        }
        
        // Handle different response formats from Anvil API
        const txHash = submitResult.txHash || submitResult.hash || submitResult.transactionHash;
        if (!txHash) {
            console.error('Submit result:', submitResult);
            throw new Error('Transaction submission failed - no transaction hash returned');
        }
        
        if (cardanoCheckout.debug) {
            console.log('Transaction hash received:', txHash);
        }
        
        // Step 5: Save order with transaction hash
        showStatus('Saving order...');
        const orderResult = await saveOrderWithPayment(txHash);
        if (cardanoCheckout.debug) {
            console.log('Order saved:', orderResult);
        }
        
        // Success!
        showPaymentSuccess(txHash);
        
    } catch (error) {
        console.error('Payment failed:', error);
        showStatus('Payment failed: ' + error.message, 'error');
    } finally {
        isProcessing = false;
        // Reset button
        if (submitBtn) {
            submitBtn.textContent = 'PLACE ORDER';
            submitBtn.disabled = false;
        }
    }
}

/**
 * Connect to Cardano wallet using CIP-30 with CardanoPress integration
 */
async function connectWallet() {
    // First, try to detect CardanoPress wallet
    const cardanoPressWallet = detectCardanoPressWallet();
    
    if (cardanoPressWallet) {
        console.log('Using CardanoPress connected wallet:', cardanoPressWallet.displayName);
        
        // Try to connect to the CardanoPress wallet
        try {
            const walletAPI = await window.cardano[cardanoPressWallet.name].enable();
            
            // Get wallet information
            const changeAddress = await walletAPI.getChangeAddress();
            const stakeAddress = await walletAPI.getRewardAddresses();
            const balance = await walletAPI.getBalance();
            
            // Ensure we have a proper Cardano address format
            let properChangeAddress = changeAddress;
            if (changeAddress && !changeAddress.startsWith('addr1') && !changeAddress.startsWith('addr_test1')) {
                try {
                    const usedAddresses = await walletAPI.getUsedAddresses();
                    const unusedAddresses = await walletAPI.getUnusedAddresses();
                    
                    if (usedAddresses && usedAddresses.length > 0) {
                        properChangeAddress = usedAddresses[0];
                    } else if (unusedAddresses && unusedAddresses.length > 0) {
                        properChangeAddress = unusedAddresses[0];
                    }
                } catch (addrError) {
                    console.warn('Could not get proper address format, using original:', addrError);
                }
            }
            
            // Create display address for UI (human-readable)
            let displayAddress = properChangeAddress;
            if (properChangeAddress && (properChangeAddress.startsWith('01') || properChangeAddress.startsWith('0x01'))) {
                // For CBOR addresses, create a truncated display version
                const cleanAddress = properChangeAddress.startsWith('0x') ? properChangeAddress.slice(2) : properChangeAddress;
                displayAddress = `addr1${cleanAddress.substring(0, 20)}...${cleanAddress.substring(cleanAddress.length - 20)}`;
            }
            
            // Create wallet object
            wallet = {
                name: cardanoPressWallet.name,
                displayName: cardanoPressWallet.displayName,
                changeAddress: properChangeAddress, // Raw address for backend
                displayAddress: displayAddress,     // Human-readable for UI
                stakeAddress: stakeAddress[0],
                balance: balance,
                api: walletAPI,
                isCardanoPress: true,
                signTx: async (tx) => {
                    return await walletAPI.signTx(tx, true);
                }
            };
            
            // Save wallet choice for future use
            saveWalletChoice(wallet);
            
            if (cardanoCheckout.debug) {
                console.log('CardanoPress wallet connected:', wallet.changeAddress);
                console.log('Wallet balance:', wallet.balance);
            }
            
            return; // Successfully connected to CardanoPress wallet
            
        } catch (error) {
            console.warn('Failed to connect to CardanoPress wallet, falling back to available wallets:', error);
        }
    }
    
    // Fallback to regular wallet detection
    const availableWallets = detectAvailableWallets();
    
    if (availableWallets.length === 0) {
        throw new Error('No Cardano wallets found. Please install Eternl, Lace, or another CIP-30 compatible wallet.');
    }
    
    // Try to connect to the first available wallet
    const selectedWallet = availableWallets[0];
    
    try {
        // Enable the wallet
        const walletAPI = await selectedWallet.api.enable();
        
        // Get wallet information
        const changeAddress = await walletAPI.getChangeAddress();
        const stakeAddress = await walletAPI.getRewardAddresses();
        const balance = await walletAPI.getBalance();
        
        // Ensure we have a proper Cardano address format
        // Some wallets return raw key hashes, we need to convert them
        let properChangeAddress = changeAddress;
        if (changeAddress && !changeAddress.startsWith('addr1') && !changeAddress.startsWith('addr_test1')) {
            // This is likely a raw key hash, we need to get the proper address
            try {
                // Try to get used addresses first, then unused addresses
                const usedAddresses = await walletAPI.getUsedAddresses();
                const unusedAddresses = await walletAPI.getUnusedAddresses();
                
                if (usedAddresses && usedAddresses.length > 0) {
                    properChangeAddress = usedAddresses[0];
                } else if (unusedAddresses && unusedAddresses.length > 0) {
                    properChangeAddress = unusedAddresses[0];
                }
                
                if (cardanoCheckout.debug) {
            console.log('Converted address from', changeAddress, 'to', properChangeAddress);
        }
            } catch (addrError) {
                console.warn('Could not get proper address format, using original:', addrError);
            }
        }
        
        // Create display address for UI (human-readable)
        let displayAddress = properChangeAddress;
        if (properChangeAddress && (properChangeAddress.startsWith('01') || properChangeAddress.startsWith('0x01'))) {
            // For CBOR addresses, create a truncated display version
            const cleanAddress = properChangeAddress.startsWith('0x') ? properChangeAddress.slice(2) : properChangeAddress;
            displayAddress = `addr1${cleanAddress.substring(0, 20)}...${cleanAddress.substring(cleanAddress.length - 20)}`;
        }
        
        // Create wallet object compatible with existing code
        wallet = {
            name: selectedWallet.name,
            displayName: selectedWallet.displayName,
            changeAddress: properChangeAddress, // Raw address for backend
            displayAddress: displayAddress,     // Human-readable for UI
            stakeAddress: stakeAddress[0],
            balance: balance,
            api: walletAPI,
            isCardanoPress: false,
            // Add methods for transaction signing
            signTx: async (tx) => {
                return await walletAPI.signTx(tx, true);
            }
        };
        
        // Save wallet choice for future use
        saveWalletChoice(wallet);
        
        if (cardanoCheckout.debug) {
            console.log('Wallet connected:', wallet.changeAddress);
            console.log('Wallet balance:', wallet.balance);
        }
        
    } catch (error) {
        throw new Error('Failed to connect to wallet: ' + error.message);
    }
}

/**
 * Build payment transaction via WordPress AJAX
 */
async function buildPaymentTransaction() {
    const formData = new FormData();
    formData.append('action', 'cardanobuildtransaction');
    formData.append('nonce', cardanoCheckout.nonce);
    formData.append('merchantaddress', productData.merchantAddress);
    formData.append('customeraddress', wallet.changeAddress);
    
    const quantity = parseInt(document.querySelector('[name="quantity"]').value) || 1;
    const totalPrice = productData.productPrice * quantity;
    
    formData.append('usdprice', totalPrice);
    formData.append('productname', productData.productName);
    
    // Debug logging - log all form data being sent (only in development)
    if (cardanoCheckout.debug) {
        console.log('=== AJAX Request Debug ===');
        console.log('Action:', 'cardanobuildtransaction');
        console.log('Nonce:', cardanoCheckout.nonce);
        console.log('Merchant Address:', productData.merchantAddress);
        console.log('Customer Address:', wallet.changeAddress);
        console.log('Quantity:', quantity);
        console.log('Total Price:', totalPrice);
        console.log('Product Name:', productData.productName);
        console.log('Product Data Object:', productData);
        console.log('Wallet Object:', wallet);
        console.log('========================');
    }
    
    // Log form data entries (only in development)
    if (cardanoCheckout.debug) {
        console.log('FormData entries:');
        for (let [key, value] of formData.entries()) {
            console.log(`${key}: ${value}`);
        }
    }
    
    const response = await fetch(cardanoCheckout.ajaxurl, {
        method: 'POST',
        body: formData
    });
    
    const result = await response.json();
    
    // Debug logging - log server response (only in development)
    if (cardanoCheckout.debug) {
        console.log('=== Server Response Debug ===');
        console.log('Response Status:', response.status);
        console.log('Response OK:', response.ok);
        console.log('Full Response:', result);
        console.log('Response Data:', result.data);
        console.log('Transaction Object:', result.data.transaction);
        console.log('Transaction Type:', typeof result.data.transaction);
        console.log('Anvil API Response Structure:', JSON.stringify(result.data.transaction, null, 2));
        console.log('============================');
    }
    
    if (!result.success) {
        console.error('Server Error Details:', result.data);
        // Check if it's a rate limiting error
        if (result.data?.message && result.data.message.includes('Too many requests')) {
            throw new Error('Please wait a moment before trying again. Too many requests detected.');
        }
        throw new Error(result.data?.message || 'Failed to build transaction');
    }
    
    return result.data;
}

/**
 * Submit transaction via WordPress AJAX
 */
async function submitPaymentTransaction(transaction, signatures) {
    if (cardanoCheckout.debug) {
        console.log('=== Submitting Payment Transaction ===');
        console.log('Transaction object:', transaction);
        console.log('Signatures:', signatures);
    }
    
    // Extract the raw transaction string (complete field) for submission
    const rawTransaction = transaction.complete || transaction;
    if (cardanoCheckout.debug) {
        console.log('Raw transaction for submission:', rawTransaction.substring(0, 100) + '...');
    }
    
    const formData = new FormData();
    formData.append('action', 'cardanosubmittransaction');
    formData.append('nonce', cardanoCheckout.nonce);
    formData.append('transaction', rawTransaction); // Send raw string, not JSON object
    formData.append('signatures', JSON.stringify(signatures));
    
    if (cardanoCheckout.debug) {
        console.log('FormData entries:');
        for (let [key, value] of formData.entries()) {
            console.log(key + ':', value);
        }
    }
    
    const response = await fetch(cardanoCheckout.ajaxurl, {
        method: 'POST',
        body: formData
    });
    
    if (cardanoCheckout.debug) {
        console.log('Submit transaction response status:', response.status);
    }
    const result = await response.json();
    if (cardanoCheckout.debug) {
        console.log('Submit transaction result:', result);
    }
    
    if (!result.success) {
        console.error('Transaction submission failed:', result.data);
        // Check if it's a rate limiting error
        if (result.data?.message && result.data.message.includes('Too many requests')) {
            throw new Error('Please wait a moment before trying again. Too many requests detected.');
        }
        throw new Error(result.data?.message || 'Failed to submit transaction');
    }
    
    if (cardanoCheckout.debug) {
        console.log('Transaction submitted successfully:', result.data);
    }
    return result.data;
}

/**
 * Save order with payment verification
 */
async function saveOrderWithPayment(txHash) {
    if (cardanoCheckout.debug) {
        console.log('=== Saving Order with Payment ===');
        console.log('Transaction Hash:', txHash);
        console.log('Product Data:', productData);
    }
    
    // Collect form data
    const formData = new FormData(document.getElementById('cardano-checkout-form'));
    const orderData = {};
    
    for (let [key, value] of formData.entries()) {
        orderData[key] = value;
    }
    
    // Add product details and transaction hash
    orderData.product_id = productData.productId;
    orderData.product_name = productData.productName;
    orderData.tx_hash = txHash;
    
    // Calculate totals
    const quantity = parseInt(orderData.quantity) || 1;
    const unitPrice = productData.productPrice;
    orderData.order_total_usd = (unitPrice * quantity).toFixed(2);
    orderData.order_total_ada = ((unitPrice * quantity) / adaPrice).toFixed(6);
    
    if (cardanoCheckout.debug) {
        console.log('Order Data to Submit:', orderData);
    }
    
    // Submit order with payment proof
    const response = await fetch(cardanoCheckout.ajaxurl, {
        method: 'POST',
        headers: {
            'Content-Type': 'application/x-www-form-urlencoded',
        },
        body: new URLSearchParams({
            action: 'cardano_submit_order',
            nonce: cardanoCheckout.nonce,
            ...orderData
        })
    });
    
    if (cardanoCheckout.debug) {
        console.log('Order submission response status:', response.status);
    }
    const result = await response.json();
    if (cardanoCheckout.debug) {
        console.log('Order submission result:', result);
    }
    
    if (!result.success) {
        console.error('Order submission failed:', result.data);
        // Check if it's a rate limiting error
        if (result.data?.message && result.data.message.includes('Too many requests')) {
            throw new Error('Please wait a moment before trying again. Too many requests detected.');
        }
        throw new Error(result.data?.message || 'Failed to save order');
    }
    
    if (cardanoCheckout.debug) {
        console.log('Order saved successfully:', result.data);
    }
    return result.data;
}

/**
 * Show status message
 */
function showStatus(message, type = 'info') {
    console.log('Status:', message);
    // You can add UI status display here if needed
}

/**
 * Show payment success
 */
function showPaymentSuccess(txHash) {
    const explorerUrl = getExplorerUrl(txHash);
    
    if (cardanoCheckout.debug) {
        console.log('=== Payment Success ===');
        console.log('Transaction Hash:', txHash);
        console.log('Explorer URL:', explorerUrl);
    }
    
    // Close modal first
    closeCheckoutModal();
    
    // Redirect to confirmation page immediately (use absolute URL)
    const redirectUrl = cardanoCheckout.home_url + '/product-order-submitted/';
    
    if (cardanoCheckout.debug) {
        console.log('Redirecting to:', redirectUrl);
    }
    
    // Redirect immediately without popup
    window.location.href = redirectUrl;
}


/**
 * Get blockchain explorer URL
 */
function getExplorerUrl(txHash) {
    const network = cardanoCheckout.network || 'preprod';
    
    if (network === 'mainnet') {
        return `https://cardanoscan.io/transaction/${txHash}`;
    } else {
        return `https://${network}.cardanoscan.io/transaction/${txHash}`;
    }
}

function resetForm() {
    // Reset form fields
    const form = document.getElementById('cardano-checkout-form');
    if (form) {
        form.reset();
    }
    
    // Reset step
    currentStep = 1;
    showStep(1);
}

// Real-time validation functions
function initializeRealTimeValidation() {
    // Email validation
    const emailField = document.getElementById('email');
    if (emailField) {
        emailField.addEventListener('blur', validateEmail);
        emailField.addEventListener('input', clearFieldError);
    }

    // Phone validation
    const phoneField = document.getElementById('phone');
    if (phoneField) {
        phoneField.addEventListener('blur', validatePhone);
        phoneField.addEventListener('input', clearFieldError);
    }

    // Address validation
    const addressFields = ['shipping_address_1', 'city', 'state_province', 'postal_code', 'country'];
    addressFields.forEach(fieldId => {
        const field = document.getElementById(fieldId);
        if (field) {
            field.addEventListener('blur', () => validateAddress(fieldId));
            field.addEventListener('input', clearFieldError);
        }
    });
}

function validateEmail() {
    const emailField = document.getElementById('email');
    const email = emailField.value.trim();
    const emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
    
    if (email && !emailRegex.test(email)) {
        showFieldError(emailField, 'Please enter a valid email address');
        return false;
    }
    return true;
}

function validatePhone() {
    const phoneField = document.getElementById('phone');
    const phone = phoneField.value.trim();
    // Basic phone validation - at least 10 digits, can include +, -, spaces, parentheses
    const phoneRegex = /^[\+]?[1-9][\d]{0,15}$/;
    
    if (phone && !phoneRegex.test(phone.replace(/[\s\-\(\)]/g, ''))) {
        showFieldError(phoneField, 'Please enter a valid phone number with country code');
        return false;
    }
    return true;
}

function validateAddress(fieldId) {
    const field = document.getElementById(fieldId);
    const value = field.value.trim();
    
    if (value.length < 2) {
        showFieldError(field, 'This field must be at least 2 characters long');
        return false;
    }
    return true;
}

function showFieldError(field, message) {
    clearFieldError(field);
    field.style.borderColor = '#dc2626';
    
    const errorDiv = document.createElement('div');
    errorDiv.className = 'field-error';
    errorDiv.style.color = '#dc2626';
    errorDiv.style.fontSize = '12px';
    errorDiv.style.marginTop = '4px';
    errorDiv.textContent = message;
    
    field.parentNode.appendChild(errorDiv);
}

function clearFieldError(field) {
    if (!field || !field.style) return;
    field.style.borderColor = '#e5e7eb';
    const existingError = field.parentNode ? field.parentNode.querySelector('.field-error') : null;
    if (existingError) {
        existingError.remove();
    }
}

// Enhanced validation for step progression
function validateCurrentStep() {
    const currentStepElement = document.getElementById('step-' + currentStep);
    if (!currentStepElement) return false;

    let isValid = true;

    if (currentStep === 1) {
        // Contact details validation
        isValid = validateEmail() && validatePhone();
        
        const requiredFields = currentStepElement.querySelectorAll('input[required]');
        requiredFields.forEach(field => {
            if (!field.value.trim()) {
                showFieldError(field, 'This field is required');
                isValid = false;
            }
        });
    } else if (currentStep === 2) {
        // Shipping details validation
        const addressFields = ['shipping_address_1', 'city', 'state_province', 'postal_code', 'country'];
        addressFields.forEach(fieldId => {
            if (!validateAddress(fieldId)) {
                isValid = false;
            }
        });
    }

    if (!isValid) {
        alert('Please correct the highlighted fields before continuing.');
    }

    return isValid;
}

/**
 * Initialize field styling for filled/empty states
 */
function initializeFieldStyling() {
    // Get all form inputs
    const inputs = document.querySelectorAll('#cardano-checkout-form input, #cardano-checkout-form textarea, #cardano-checkout-form select');
    
    inputs.forEach(input => {
        // Add event listeners for focus/blur/input
        input.addEventListener('focus', function() {
            this.style.borderColor = '#2DB0B8';
        });
        
        input.addEventListener('blur', function() {
            if (this.value.trim() !== '') {
                this.classList.add('filled');
                this.style.borderColor = '#2DB0B8';
            } else {
                this.classList.remove('filled');
                this.style.borderColor = '#333';
            }
        });
        
        input.addEventListener('input', function() {
            if (this.value.trim() !== '') {
                this.classList.add('filled');
                this.style.borderColor = '#2DB0B8';
            } else {
                this.classList.remove('filled');
                this.style.borderColor = '#333';
            }
        });
        
        // Check initial state
        if (input.value.trim() !== '') {
            input.classList.add('filled');
            input.style.borderColor = '#2DB0B8';
        }
    });
}

// Make functions globally available
window.nextStep = nextStep;
window.prevStep = prevStep;
window.updateOrderTotal = updateOrderTotal;
